<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Admin\AssignmentDetails;
use App\Models\Admin\AssignmentFiles;
use App\Models\Admin\Assignments;
use App\Models\Admin\Template;
use App\Models\Admin\AssignmentWhyChoose;
use App\Models\Admin\Coupons;
use App\Models\Admin\EducationalSessions;
use App\Models\Admin\Medium;
use App\Models\Admin\Subjects;
use App\Models\Admin\Types;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Yajra\DataTables\Facades\DataTables;

class AssignmentsController extends Controller
{
    private function getNextFileIndex($assignmentId, $langId, $sessId)
    {
        return AssignmentFiles::where('assignment_id', $assignmentId)
            ->where('language_id', $langId)
            ->where('session_id', $sessId)
            ->max('file_index') + 1;
    }

    public function checkAssignmentLink($assignment_url, $encrypted_id = "")
    {
        $assignment_url = str_replace(['/', ' '], '-', $assignment_url);
        $assignment_url = preg_replace('/[^a-z0-9-]+/', '-', $assignment_url);
        $assignment_url = trim($assignment_url, '-');
        $assignment_url = preg_replace('/-+/', '-', $assignment_url);
        $original_link = $assignment_url;

        $id = (isset($encrypted_id) && !empty($encrypted_id)) ? Crypt::decrypt($encrypted_id) : 0;
        $suffix = 1;
        do {
            $count = $id != 0 ? Assignments::where('assignment_url', $assignment_url)->where('id', '!=', $id)->count() : Assignments::where('assignment_url', $assignment_url)->count();

            if ($count > 0) {
                $assignment_url = $original_link . '-' . $suffix;
                $suffix++;
            } else {
                break;
            }
        } while (true);

        return $assignment_url;
    }

    public function manageAssignments(Request $request)
    {
        $main_page = 'assignments_management';
        $currentPage = "manage_assignments";
        return view('admin.manage_assignments', ['courseId' => $request->course, 'main_page' => $main_page, 'currentPage' => $currentPage]);
    }

    public function manageAssignmentsData(Request $request)
    {
        $assignments = Assignments::with('subject')
            ->select(
                'id',
                'subject_id',
                'position_order',
                'assignment_type',
                'assignment_code',
                'assignment_title',
                'assignment_image',
                'status'
            )->orderBy('position_order')->get();

        $model = Crypt::encrypt('Assignments');

        $data = [];
        $sno = 1;
        foreach ($assignments as $assignment) {
            $assignment->encrypted_id = Crypt::encrypt($assignment->id);
            $payload = compact('model', 'assignment', 'sno');
            $data[] = [
                'checkbox' => '<input type="checkbox" class="row_checkbox" value="' . $assignment->encrypted_id . '">',
                'id' => $sno++,
                'subject' => $assignment->subject->subject_code . ' (' . $assignment->subject->subject_name . ')',
                'assignment_type' => $assignment->assignment_type == 'solved' ? 'Solved PDF' : 'Hand-Written',
                'assignment_code' => '<strong>' . $assignment->assignment_code . '</strong>',
                'assignment_title' => $assignment->assignment_title,
                'assignment_image' => '<ul class="list-unstyled users-list m-0 avatar-group d-flex align-items-center">
                    <li data-bs-toggle="tooltip" data-popup="tooltip-custom" data-bs-placement="top" class="avatar avatar-md pull-up" title="' . $assignment->assignment_title . '" >
                        <img src="' . asset($assignment->assignment_image) . '" class="rounded-circle" width="40" height="40">
                    </li>
                </ul>',
                'position_order' => $assignment->position_order,
                'status' => view('admin.components.status', $payload)->render(),
                'action' => view('admin.components.action', $payload)->render(),
            ];
        }
        return DataTables::of($data)->rawColumns(['checkbox', 'assignment_code', 'assignment_image', 'status', 'action'])->make(true);
    }

    public function bulkStatus(Request $request)
    {
        $request->validate([
            'encrypted_ids' => 'required|array',
            'encrypted_ids.*' => 'required|string',
        ]);

        try {
            $ids = collect($request->encrypted_ids)->map(function ($encrypted_id) {
                return Crypt::decrypt($encrypted_id);
            })->filter()->values();

            if ($ids->isEmpty()) {
                return response()->json([
                    'status'  => 'error',
                    'title'   => 'Invalid selection',
                    'message' => 'No valid assignments selected.'
                ], 422);
            }

            $status_updated = Assignments::whereIn('id', $ids)
                ->update([
                    'status' => DB::raw("
                        CASE
                            WHEN status = 'active' THEN 'deactive'
                            WHEN status = 'deactive' THEN 'active'
                            ELSE status
                        END
                    ")
                ]);

            if ($status_updated > 0) {
                return response()->json(['status' => 'done', 'title' => "Status operation is done for selected Assignments.", 'message' => "Your files have been updated."], 200);
            }

            return response()->json([
                'status'  => 'error',
                'title' => "Failed to update!",
                'message' => "Your data is safe :)"
            ], 404);
        } catch (\Exception $e) {

            Log::error('Bulk update exception occurred', [
                'error'     => $e->getMessage(),
                'trace'     => $e->getTraceAsString(),
                'ip'        => $request->ip(),
            ]);

            return response()->json([
                'status'  => 'error',
                'title'   => 'Update Failed',
                'message' => 'Something went wrong while updating assignments.'
            ], 500);
        }
    }

    public function bulkDelete(Request $request)
    {
        $request->validate([
            'encrypted_ids' => 'required|array',
            'encrypted_ids.*' => 'required|string',
        ]);

        try {
            $ids = collect($request->encrypted_ids)->map(function ($encrypted_id) {
                return Crypt::decrypt($encrypted_id);
            })->filter()->values();

            if ($ids->isEmpty()) {
                return response()->json([
                    'status'  => 'error',
                    'title'   => 'Invalid selection',
                    'message' => 'No valid assignments selected.'
                ], 422);
            }

            $deleted = Assignments::whereIn('id', $ids)->delete();

            if ($deleted > 0) {
                return response()->json(['status' => 'done', 'title' => "Delete operation is done for selected Assignments.", 'message' => "Your files have been deleted."], 200);
            }

            return response()->json([
                'status'  => 'error',
                'title' => "Failed to Delete!",
                'message' => "Your data is safe :)"
            ], 404);
        } catch (\Exception $e) {

            Log::error('Bulk delete exception occurred', [
                'error'     => $e->getMessage(),
                'trace'     => $e->getTraceAsString(),
                'ip'        => $request->ip(),
            ]);

            return response()->json([
                'status'  => 'error',
                'title'   => 'Delete Failed',
                'message' => 'Something went wrong while deleting assignments.'
            ], 500);
        }
    }

    private function decryptArray(?array $items): array
    {
        return collect($items ?? [])
            ->map(fn($id) => Crypt::decrypt($id))
            ->all();
    }

    private function safe(?string $text): ?string
    {
        return $text ? htmlspecialchars($text, ENT_QUOTES) : null;
    }

    private function uploadFile(Request $request, string $field, string $path): ?string
    {
        if (!$request->hasFile($field)) {
            return null;
        }

        return $this->storeImage($request->file($field), $path);
    }

    private function getDynamicDescription($subject_id, $languageIds, $sessionIds, $template_id, $code)
    {
        $subject = Subjects::findOrFail($subject_id);

        $languages = Medium::whereIn('id', $languageIds)
            ->pluck('language_name')
            ->implode(', ');

        $sessions = EducationalSessions::whereIn('id', $sessionIds)
            ->pluck('session_name')
            ->implode(', ');

        $template = Template::find($template_id);
        $short_description = $template->short_description;
        $download_info = $template->download_info;
        $description = $template->description;
        $cover_page_info = $template->cover_page_info;
        // $terms_conditions_text = $template->terms_conditions_text;

        $short_description = str_replace(
            [
                '{ CODE }',
                '{ SUBJECT_NAME }',
                '{ SESSIONS }',
                '{ LANGUAGES }',
            ],
            [
                $code,
                $subject->subject_name,
                $sessions,
                $languages,
            ],
            $short_description
        );

        $download_info = str_replace(
            [
                '{ CODE }',
                '{ SUBJECT_NAME }',
                '{ SESSIONS }',
                '{ LANGUAGES }',
            ],
            [
                $code,
                $subject->subject_name,
                $sessions,
                $languages,
            ],
            $download_info
        );

        $description = str_replace(
            [
                '{ CODE }',
                '{ SUBJECT_NAME }',
                '{ SESSIONS }',
                '{ LANGUAGES }',
            ],
            [
                $code,
                $subject->subject_name,
                $sessions,
                $languages,
            ],
            $description
        );

        $cover_page_info = str_replace(
            [
                '{ CODE }',
                '{ SUBJECT_NAME }',
                '{ SESSIONS }',
                '{ LANGUAGES }',
            ],
            [
                $code,
                $subject->subject_name,
                $sessions,
                $languages,
            ],
            $cover_page_info
        );

        return [$short_description, $download_info, $description, $cover_page_info];
    }

    public function manageAssignmentTemplate(Request $request)
    {
        $assignment_template = Template::find(1);

        if (!$request->isMethod('post')) {
            return view('admin.template', [
                'assignment_template' => $assignment_template,
                'main_page' => 'assignments_management',
                'currentPage' => 'assignment_template',
            ]);
        }

        DB::beginTransaction();

        try {
            $assignment_template->update([
                'template_type' => 'assignment',
                'short_description' => $this->safe($request->short_description),
                'download_info' => $this->safe($request->download_info),
                'description' => $this->safe($request->description),
                'cover_page_info' => $this->safe($request->cover_page_info),
                // 'terms_conditions_text' => $this->safe($request->terms_conditions_text),
            ]);

            DB::commit();

            return redirect()
                ->route('manage.assignment.template')
                ->with('success', 'Assignment Template updated successfully!');
        } catch (\Throwable $e) {
            DB::rollBack();

            Log::error('Assignment template updation failed', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return redirect()
                ->route('manage.assignment.template')
                ->with('error', 'Something went wrong while updating assignment template.');
        }
    }

    public function addAssignment(Request $request)
    {
        if (!$request->isMethod('post')) {
            return $this->renderAssignmentForm($request);
        }

        $validated = $request->validate([
            'subject_id' => 'required|string',

            'type_ids' => 'array|nullable',
            'language_ids' => 'array|nullable',
            'session_ids' => 'array|nullable',
            'coupon_ids' => 'array|nullable',

            'assignment_type' => 'required|string',
            'assignment_code' => 'required|string',
            'assignment_url' => 'required|string',
            'assignment_title' => 'required|string',
            'maximum_retail_price' => 'required|numeric',
            'selling_price' => 'required|numeric',
        ]);

        DB::beginTransaction();

        try {
            $position_order = (Assignments::max('position_order') ?? 0) + 1;

            if (!empty($request->assignment_url)) {
                $link = strtolower($request->assignment_url);
            } else {
                $link = strtolower($request->assignment_code);
            }
            $assignment_url = $this->checkAssignmentLink($link);
            $subject_id = Crypt::decrypt($request->subject_id);
            $languageIds = $this->decryptArray($request->language_ids);
            $sessionIds  = $this->decryptArray($request->session_ids);

            [$short_description, $download_info, $description, $cover_page_info] = $this->getDynamicDescription($subject_id, $languageIds, $sessionIds, 1, $request->assignment_code);

            $assignment = Assignments::create([
                'subject_id' => $subject_id,
                'position_order' => $position_order,

                'type_ids' => $this->decryptArray($request->type_ids),
                'language_ids' => $languageIds,
                'session_ids' => $sessionIds,
                'coupon_ids' => $this->decryptArray($request->coupon_ids),

                'assignment_type' => $request->assignment_type,
                'assignment_code' => $request->assignment_code,
                'assignment_subject_name' => $request->assignment_subject_name,
                'assignment_url' => $assignment_url,
                'assignment_title' => $request->assignment_title,

                'breadcrumb_headline' => $request->breadcrumb_headline,
                'assignment_quantity' => $request->assignment_quantity,

                'maximum_retail_price' => $request->maximum_retail_price,
                'selling_price' => $request->selling_price,
                'discount' => $request->discount,
                'additional_discount' => $request->additional_discount,
                'discount_offer_expiry' => $request->discount_offer_expiry,
                'assignment_expiry' => $request->assignment_expiry,

                'question_paper_text' => $this->safe($request->question_paper_text),
                'short_description' => $short_description,
                'download_info' => $download_info,
                'description' => $description,
                'cover_page_info' => $cover_page_info,
                // 'terms_conditions_text' => $this->safe($request->terms_conditions_text),

                'meta_title' => $request->meta_title,
                'meta_keyword' => $request->meta_keyword,
                'meta_description' => $this->safe($request->meta_description),
            ]);

            // file uploads
            $assignment->update([
                'assignment_image' => $this->uploadFile($request, 'assignment_image', 'images/assignments/'),
                'question_paper_image' => $this->uploadFile($request, 'question_paper_image', 'images/assignments/question_paper_images/')
            ]);

            DB::commit();

            return redirect()
                ->route('manage.assignments', $request->course)
                ->with('success', 'Assignment inserted successfully!');
        } catch (\Throwable $e) {
            DB::rollBack();

            Log::error('Assignment insert failed', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return redirect()
                ->route('add.assignment', $request->course)
                ->with('error', 'Something went wrong while saving assignment.');
        }
    }

    private function renderAssignmentForm($request)
    {
        $all_assignment_types = Types::active()->get()->each(fn($i) => $i->encrypted_id = Crypt::encrypt($i->id));
        $all_assignment_languages = Medium::active()->get()->each(fn($i) => $i->encrypted_id = Crypt::encrypt($i->id));
        $all_assignment_sessions = EducationalSessions::active()->get()->each(fn($i) => $i->encrypted_id = Crypt::encrypt($i->id));
        $all_assignment_coupons = Coupons::where('reference_type', 'assignment')
            ->active()
            ->get()
            ->each(fn($i) => $i->encrypted_id = Crypt::encrypt($i->id));
        $all_subjects = Subjects::all();
        foreach ($all_subjects as $subject) {
            $subject->encrypted_id = Crypt::encrypt($subject->id);
        }

        return view('admin.assignment-ops', [
            'all_subjects' => $all_subjects,
            'all_assignment_types' => $all_assignment_types,
            'all_assignment_languages' => $all_assignment_languages,
            'all_assignment_sessions' => $all_assignment_sessions,
            'all_assignment_coupons' => $all_assignment_coupons,
            'main_page' => 'assignments_management',
            'currentPage' => 'manage_assignments',
        ]);
    }

    public function editAssignment(Request $request)
    {
        if ($request->isMethod('post')) {
            $request->validate([
                'subject_id' => 'required|string',
                'type_ids' => 'array|nullable',
                'language_ids' => 'array|nullable',
                'session_ids' => 'array|nullable',
                'coupon_ids' => 'array|nullable',
                'assignment_type' => 'required|string',
                'assignment_code' => 'required|string',
                'assignment_url' => 'required|string',
                'assignment_title' => 'required|string',
                'maximum_retail_price' => 'required|numeric',
                'selling_price' => 'required|numeric',
                'included_section_icons' => 'array|nullable',
                'included_section_titles' => 'array|nullable',
                'included_section_descriptions' => 'array|nullable',
            ], [
                'assignment_type.required' => 'Please provide a Assignment Type.',
                'assignment_type.string' => 'Assignment Type must be a string.',
                'assignment_code.required' => 'Please provide a Assignment Code.',
                'assignment_code.string' => 'Assignment Code must be a string.',
                'assignment_title.required' => 'Please provide Assignment Title.',
                'assignment_title.string' => 'Assignment Title must be a string.',
                'maximum_retail_price.required' => 'Please provide MRP.',
                'maximum_retail_price.numeric' => 'MRP must be in numeric.',
                'selling_price.required' => 'Please provide Selling Price.',
                'selling_price.numeric' => 'Selling Price must be in numeric.',
            ]);

            $type_ids = array_map(function ($id) {
                return Crypt::decrypt($id);
            }, $request->type_ids ?? []);
            $language_ids = array_map(function ($id) {
                return Crypt::decrypt($id);
            }, $request->language_ids ?? []);
            $session_ids = array_map(function ($id) {
                return Crypt::decrypt($id);
            }, $request->session_ids ?? []);
            $coupon_ids = array_map(function ($id) {
                return Crypt::decrypt($id);
            }, $request->coupon_ids ?? []);

            $included_section_descriptions = array_map(function ($desc) {
                return htmlspecialchars($desc, ENT_QUOTES);
            }, $request->included_section_descriptions ?? []);

            if (!empty($request->assignment_url)) {
                $link = strtolower($request->assignment_url);
            } else {
                $link = strtolower($request->assignment_code);
            }
            $assignment_url = $this->checkAssignmentLink($link, $request->assignment);

            $subject_id = Crypt::decrypt($request->subject_id);

            [$short_description, $download_info, $description, $cover_page_info] = $this->getDynamicDescription($subject_id, $language_ids, $session_ids, 1, $request->assignment_code);

            $id = Crypt::decrypt($request->assignment);
            $assignment = Assignments::findOrFail($id);
            $assignment->subject_id = $subject_id;
            $assignment->type_ids = $type_ids;
            $assignment->language_ids = $language_ids;
            $assignment->session_ids = $session_ids;
            $assignment->coupon_ids = $coupon_ids;
            $assignment->assignment_type = $request->assignment_type;
            $assignment->assignment_code = $request->assignment_code;
            $assignment->assignment_subject_name = $request->assignment_subject_name;
            $assignment->assignment_url = $assignment_url;
            $assignment->assignment_title = $request->assignment_title;
            $assignment->breadcrumb_headline = $request->breadcrumb_headline;
            $assignment->assignment_quantity = $request->assignment_quantity;
            $assignment->maximum_retail_price = $request->maximum_retail_price;
            $assignment->selling_price = $request->selling_price;
            $assignment->discount = $request->discount;
            $assignment->additional_discount = $request->additional_discount;
            $assignment->discount_offer_expiry = $request->discount_offer_expiry;
            $assignment->assignment_expiry = $request->assignment_expiry;
            $assignment->question_paper_text = htmlspecialchars($request->question_paper_text, ENT_QUOTES);
            $assignment->short_description = isset($request->short_description) ? htmlspecialchars($request->short_description, ENT_QUOTES) : $short_description;
            $assignment->download_info = isset($request->download_info) ? htmlspecialchars($request->download_info, ENT_QUOTES) : $download_info;
            $assignment->description = isset($request->description) ? htmlspecialchars($request->description, ENT_QUOTES) : $description;
            $assignment->cover_page_info = isset($request->cover_page_info) ? htmlspecialchars($request->cover_page_info, ENT_QUOTES) : $cover_page_info;
            // $assignment->terms_conditions_text = htmlspecialchars($request->terms_conditions_text, ENT_QUOTES);
            $assignment->meta_title = $request->meta_title;
            $assignment->meta_keyword = $request->meta_keyword;
            $assignment->meta_description = htmlspecialchars($request->meta_description, ENT_QUOTES);

            if (!empty($request->file('assignment_image'))) {
                $path = 'images/assignments/';
                $filePath = $this->storeImage($request->file("assignment_image"), $path, $assignment->assignment_image);
                $assignment->assignment_image = $filePath;
            }
            if (!empty($request->file('question_paper_image'))) {
                $path = 'images/assignments/question_paper_images/';
                $filePath = $this->storeImage($request->file("question_paper_image"), $path, $assignment->question_paper_image);
                $assignment->question_paper_image = $filePath;
            }

            if ($assignment->save()) {
                $request->session()->flash('success', 'Assignment is updated Successfully!');
                return redirect()->route('manage.assignments', $request->course);
            } else {
                $request->session()->flash('error', 'Updation Error!');
                return redirect()->route('edit.assignment', ["course" => $request->course, "assignment" => $request->assignment]);
            }
        } else {
            $all_assignment_types = Types::select('id', 'type_name')
                ->where('status', 'active')
                ->get()
                ->each(fn($item) => $item->encrypted_id = Crypt::encrypt($item->id));

            $all_assignment_languages = Medium::select('id', 'language_name')
                ->where('status', 'active')
                ->get()
                ->each(fn($item) => $item->encrypted_id = Crypt::encrypt($item->id));

            $all_assignment_sessions = EducationalSessions::select('id', 'session_name')
                ->where('status', 'active')
                ->get()
                ->each(fn($item) => $item->encrypted_id = Crypt::encrypt($item->id));

            $all_assignment_coupons = Coupons::select('id', 'coupon', 'coupon_name')
                ->where([
                    'reference_type' => 'assignment',
                    'status' => 'active'
                ])
                ->get()
                ->each(fn($item) => $item->encrypted_id = Crypt::encrypt($item->id));

            $all_subjects = Subjects::all();
            foreach ($all_subjects as $subject) {
                $subject->encrypted_id = Crypt::encrypt($subject->id);
            }

            $id = Crypt::decrypt($request->assignment);
            $assignment = Assignments::where('id', $id)->firstOrFail();
            $assignment->encrypted_id = $request->assignment;

            $included_section_icons = $assignment->included_icons;
            $included_section_titles = $assignment->included_titles;
            $included_section_descriptions = $assignment->included_descriptions;

            $main_page = 'assignments_management';
            $currentPage = "manage_assignments";
            return view('admin.assignment-ops', ['all_subjects' => $all_subjects, "all_assignment_types" => $all_assignment_types, "all_assignment_languages" => $all_assignment_languages, "all_assignment_sessions" => $all_assignment_sessions, "all_assignment_coupons" => $all_assignment_coupons, "assignment" => $assignment, 'main_page' => $main_page, 'currentPage' => $currentPage], compact('included_section_icons', 'included_section_titles', 'included_section_descriptions'));
        }
    }

    public function manageAssignmentFiles(Request $request)
    {
        try {
            $assignment_id = Crypt::decrypt($request->assignment);
        } catch (\Exception $e) {
            return back()->with('error', 'Invalid assignment reference.');
        }

        $assignment = Assignments::find($assignment_id);

        if (!$assignment) {
            return back()->with('error', 'Assignment not found.');
        }

        if ($request->isMethod('post')) {
            try {

                $request->validate([
                    'rule_ids'     => 'array|nullable',
                    'rule_ids.*'   => 'nullable|string',

                    'file_types'   => 'required|array',
                    'file_types.*' => 'required|string',

                    'file_languages'    => 'required|array',
                    'file_languages.*'  => 'required|string',

                    'file_sessions'    => 'required|array',
                    'file_sessions.*'  => 'required|string',

                    'files'        => 'array|nullable',
                    'files.*'      => 'nullable|file|max:10240', // 10MB

                    'file_prices'   => 'required|array',
                    'file_prices.*' => 'required|numeric|gt:0',

                    // 'file_passwords'   => 'array|nullable',
                    // 'file_passwords.*' => 'nullable|string|min:4',
                    // 'is_protected'     => 'array|nullable',
                ]);

                DB::beginTransaction();

                $existingMaxOrder = AssignmentFiles::where('assignment_id', $assignment_id)
                    ->max('position_order');

                $position = $existingMaxOrder ? $existingMaxOrder + 1 : 1;

                foreach ($request->file_types as $i => $typeId) {

                    $langId   = $request->file_languages[$i] ?? null;
                    $sessId   = $request->file_sessions[$i] ?? null;
                    $ruleEnc  = $request->rule_ids[$i] ?? null;
                    $file     = $request->file('files')[$i] ?? null;
                    $file_price = (float) $request->file_prices[$i];

                    $type_id = Crypt::decrypt($typeId);
                    $lang_id = Crypt::decrypt($langId);
                    $sess_id = Crypt::decrypt($sessId);

                    $file_language = Medium::select('language_name')->find($lang_id);
                    $file_session = EducationalSessions::select('session_name')->find($sess_id);

                    // $isProtected = $request->is_protected[$i] ?? 0;
                    // $password    = $request->file_passwords[$i] ?? null;

                    // if ($isProtected && empty($password)) {
                    //     throw new \Exception("Password required for protected file");
                    // }
                    // if (empty($password)) {
                    //     throw new \Exception("Password required for protected file");
                    // }

                    // -----------------------------
                    // UPDATE EXISTING
                    // -----------------------------
                    if (!empty($ruleEnc)) {

                        $ruleId = Crypt::decrypt($ruleEnc);

                        $record = AssignmentFiles::where('id', $ruleId)
                            ->where('assignment_id', $assignment_id)
                            ->first();

                        if (!$record) continue;

                        $data = [
                            'type_id'       => $type_id,
                            'language_id'   => $lang_id,
                            'session_id'    => $sess_id,
                            // 'is_protected' => $isProtected,
                        ];

                        // if ($isProtected && $password) {
                        //     $data['file_password'] = Crypt::encrypt($password);
                        // }
                        // if ($password) {
                        //     $data['file_password'] = $password;
                        // }

                        // New file uploaded?
                        if ($file) {
                            $path = 'files/assignments/';
                            $index = $record->file_index;
                            $codeSlug     = Str::upper(Str::slug($assignment->assignment_code));
                            $languageSlug = Str::ucwords(Str::slug($file_language->language_name), '-');
                            $sessionSlug  = Str::ucwords(Str::slug($file_session->session_name), '-');
                            $new_file_name = "{$codeSlug}-{$languageSlug}-{$sessionSlug}-{$index}";
                            $filePath = $this->storeImage($file, $path, "", $new_file_name);

                            $data['file_path'] = $path;
                            $data['file_name'] = $new_file_name;
                            $data['file_full_path'] = $filePath;
                            $data['file_price'] = $file_price;
                        }

                        $record->update($data);
                    }

                    // -----------------------------
                    // INSERT NEW
                    // -----------------------------
                    else {

                        if (!$file) {
                            throw new \Exception("File is required for new row");
                        }

                        $path = 'files/assignments/';
                        $index = $this->getNextFileIndex($assignment_id, $lang_id, $sess_id);
                        $codeSlug     = Str::upper(Str::slug($assignment->assignment_code));
                        $languageSlug = Str::ucwords(Str::slug($file_language->language_name), '-');
                        $sessionSlug  = Str::ucwords(Str::slug($file_session->session_name), '-');
                        $new_file_name = "{$codeSlug}-{$languageSlug}-{$sessionSlug}-{$index}";
                        $filePath = $this->storeImage($file, $path, "", $new_file_name);

                        AssignmentFiles::create([
                            'assignment_id'     => $assignment_id,
                            'type_id'           => $type_id,
                            'language_id'       => $lang_id,
                            'session_id'        => $sess_id,
                            'position_order'    => $position++,
                            'file_path'         => $path,
                            'file_name'         => $new_file_name,
                            'file_index'         => $index,
                            'file_full_path'    => $filePath,
                            'file_price'        => $file_price,
                            // 'is_protected'      => $isProtected,
                            // 'file_password'     => $isProtected && $password ? Crypt::encrypt($password) : null,
                            // 'file_password'     => $password,
                        ]);
                    }
                }

                DB::commit();

                return redirect()
                    ->route('manage.assignment.files', $request->assignment)
                    ->with('success', 'Assignment files saved successfully!');
            } catch (\Throwable $e) {

                DB::rollBack();

                Log::error('Assignment files save failed', [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString(),
                ]);

                return redirect()
                    ->back()
                    ->withInput()
                    ->with('error', 'Something went wrong while saving files.');
            }
        }

        $all_assignment_types = Types::select('id', 'type_name')
            ->where('status', 'active')
            ->whereIn('id', $assignment->type_ids)
            ->get()
            ->each(fn($item) => $item->encrypted_id = Crypt::encrypt($item->id));

        $all_assignment_languages = Medium::select('id', 'language_name')
            ->where('status', 'active')
            ->whereIn('id', $assignment->language_ids)
            ->get()
            ->each(fn($item) => $item->encrypted_id = Crypt::encrypt($item->id));

        $all_assignment_sessions = EducationalSessions::select('id', 'session_name')
            ->where('status', 'active')
            ->whereIn('id', $assignment->session_ids)
            ->get()
            ->each(fn($item) => $item->encrypted_id = Crypt::encrypt($item->id));

        $main_page  = 'assignments_management';
        $currentPage = 'manage_assignments';

        return view('admin.assignment_files', [
            'assignmentId' => $request->assignment,
            'assignment' => $assignment,
            'types' => $all_assignment_types,
            'languages' => $all_assignment_languages,
            'sessions' => $all_assignment_sessions,
            'main_page'    => $main_page,
            'currentPage' => $currentPage,
        ]);
    }

    public function getAssignmentFiles(Request $request)
    {
        $assignment_id = Crypt::decrypt($request->assignment);
        $adAssignmentFiles = AssignmentFiles::select('id', 'type_id', 'language_id', 'session_id', 'file_name', 'file_full_path', 'file_price', 'file_password', 'status')->where('assignment_id', $assignment_id)->get();

        $all_assignmentFiles = $adAssignmentFiles->map(function ($adAssignmentFile) {
            return [
                'rule_id' => Crypt::encrypt($adAssignmentFile->id),
                'file_type_id' => $adAssignmentFile->type_id,
                'language_id' => $adAssignmentFile->language_id,
                'session_id' => $adAssignmentFile->session_id,
                'file_name' => $adAssignmentFile->file_name,
                'file_url' => asset($adAssignmentFile->file_full_path),
                'price' => $adAssignmentFile->file_price,
                'file_password' => $adAssignmentFile->file_password,
            ];
        });

        return response()->json($all_assignmentFiles);
    }

    public function deleteAssignmentFile(Request $request)
    {
        $rule_id = Crypt::decrypt($request->rule_id);
        $assignment_file = AssignmentFiles::find($rule_id);

        if (file_exists($assignment_file->file_full_path)) {
            $this->removeImage($assignment_file->file_full_path);
        }

        if ($assignment_file->delete()) {
            return 1;
        } else {
            return 0;
        }
    }

    public function manageAssignmentWhyChoose(Request $request)
    {
        try {
            $assignment_id = Crypt::decrypt($request->assignment);
        } catch (\Exception $e) {
            return back()->with('error', 'Invalid assignment reference.');
        }

        $assignment = Assignments::find($assignment_id);

        if (!$assignment) {
            return back()->with('error', 'Assignment not found.');
        }

        if ($request->isMethod('post')) {
            try {
                $request->validate([
                    'choose_ids'        => 'array|nullable',
                    'choose_ids.*'      => 'nullable|string',
                    'choose_contents'   => 'array|nullable',
                    'choose_contents.*' => 'nullable|string',
                ]);

                DB::beginTransaction();

                // SAFE update
                $assignment->why_choose_headline = $request->why_choose_headline;
                $assignment->save();

                // Ensure at least one content exists
                $contents = collect($request->choose_contents)->filter();
                if ($contents->isEmpty() && empty($request->choose_ids)) {
                    DB::rollBack();
                    return redirect()
                        ->route('manage.why.choose.content', $request->assignment)
                        ->with('error', 'At least one item is required!');
                }

                $existingMaxOrder = AssignmentWhyChoose::where('assignment_id', $assignment_id)
                    ->max('position_order');

                $position = $existingMaxOrder ? $existingMaxOrder + 1 : 1;

                foreach ($request->choose_contents ?? [] as $i => $content) {
                    $content = trim($content);

                    if ($content === '') {
                        continue;
                    }

                    // UPDATE EXISTING
                    if (!empty($request->choose_ids[$i])) {
                        $whyChooseId = Crypt::decrypt($request->choose_ids[$i]);
                        $whyChoose   = AssignmentWhyChoose::where('id', $whyChooseId)
                            ->where('assignment_id', $assignment_id)
                            ->first();

                        if ($whyChoose) {
                            $whyChoose->update([
                                'assignment_content' => $content,
                            ]);
                        }
                    }
                    // INSERT NEW
                    else {
                        AssignmentWhyChoose::create([
                            'assignment_id'      => $assignment_id,
                            'position_order'     => $position++,
                            'assignment_content' => $content,
                        ]);
                    }
                }

                DB::commit();

                return redirect()
                    ->route('manage.why.choose.content', $request->assignment)
                    ->with('success', 'Why Choose content saved successfully!');
            } catch (\Throwable $e) {
                DB::rollBack();

                Log::error('Why Choose save failed', [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString(),
                ]);

                return redirect()
                    ->back()
                    ->withInput()
                    ->with('error', 'Something went wrong while saving data.');
            }
        }

        $main_page  = 'assignments_management';
        $currentPage = 'manage_assignments';

        return view('admin.assignment_why_choose_content', [
            'assignmentId' => $request->assignment,
            'whyChooseHeadline' => $assignment->why_choose_headline,
            'main_page'    => $main_page,
            'currentPage' => $currentPage,
        ]);
    }

    public function getAssignmentWhyChoose(Request $request)
    {
        $assignment_id = Crypt::decrypt($request->assignment);
        $adwhychooses = AssignmentWhyChoose::select('id', 'assignment_content', 'status')->where('assignment_id', $assignment_id)->get();

        $all_whychooses = $adwhychooses->map(function ($adwhychoose) {
            return [
                'choose_id' => Crypt::encrypt($adwhychoose->id),
                'choose_content' => $adwhychoose->assignment_content,
            ];
        });

        return response()->json($all_whychooses);
    }

    public function deleteWhyChoose(Request $request)
    {
        $choose_id = Crypt::decrypt($request->choose);
        $assignment_why_choose = AssignmentWhyChoose::find($choose_id);
        if ($assignment_why_choose->delete()) {
            return 1;
        } else {
            return 0;
        }
    }

    public function manageAssignmentDetails(Request $request)
    {
        if ($request->isMethod('post')) {
            try {
                $request->validate([
                    'detail_ids'         => 'array|nullable',
                    'detail_ids.*'       => 'nullable|string',
                    'detail_titles'      => 'array|nullable',
                    'detail_titles.*'    => 'nullable|string',
                    'detail_texts'       => 'array|nullable',
                    'detail_texts.*'     => 'nullable|string',
                ]);

                DB::beginTransaction();

                // SAFE decrypt
                try {
                    $assignment_id = Crypt::decrypt($request->assignment);
                } catch (\Exception $e) {
                    return back()->with('error', 'Invalid assignment reference.');
                }

                // Ensure at least one detail exists
                $titles = collect($request->detail_titles)->filter(fn($v) => trim($v) !== '');
                if ($titles->isEmpty()) {
                    DB::rollBack();
                    return redirect()
                        ->route('manage.assignment.details.contents', $request->assignment)
                        ->with('error', 'At least one detail title is required!');
                }

                // Get current max position
                $existingMaxOrder = AssignmentDetails::where('assignment_id', $assignment_id)
                    ->max('position_order');

                $position = $existingMaxOrder ? $existingMaxOrder + 1 : 1;

                foreach ($request->detail_titles ?? [] as $i => $title) {
                    $title = trim($title);
                    $text  = trim($request->detail_texts[$i] ?? '');

                    if ($title === '') {
                        continue;
                    }

                    $detailId = $request->detail_ids[$i] ?? null;

                    // UPDATE EXISTING
                    if (!empty($detailId)) {
                        try {
                            $decodedId = Crypt::decrypt($detailId);
                        } catch (\Exception $e) {
                            continue;
                        }

                        $detail = AssignmentDetails::where('id', $decodedId)
                            ->where('assignment_id', $assignment_id)
                            ->first();

                        if ($detail) {
                            $detail->update([
                                'assignment_detail_title' => $title,
                                'assignment_detail_text'  => $text,
                            ]);
                        }
                    }
                    // INSERT NEW
                    else {
                        AssignmentDetails::create([
                            'assignment_id'            => $assignment_id,
                            'position_order'           => $position++,
                            'assignment_detail_title'  => $title,
                            'assignment_detail_text'   => $text,
                        ]);
                    }
                }

                DB::commit();

                return redirect()
                    ->route('manage.assignment.details.contents', $request->assignment)
                    ->with('success', 'Assignment details saved successfully!');
            } catch (\Throwable $e) {
                DB::rollBack();

                Log::error('Assignment Details save failed', [
                    'error' => $e->getMessage(),
                    'file'  => $e->getFile(),
                    'line'  => $e->getLine(),
                ]);

                return back()
                    ->withInput()
                    ->with('error', 'Something went wrong while saving details.');
            }
        }

        return view('admin.assignment_other_details', [
            'assignmentId' => $request->assignment,
            'main_page'    => 'assignments_management',
            'currentPage' => 'manage_assignments',
        ]);
    }

    public function getAssignmentDetails(Request $request)
    {
        $assignment_id = Crypt::decrypt($request->assignment);
        $ad_details = AssignmentDetails::select('id', 'assignment_detail_title', 'assignment_detail_text', 'status')->where('assignment_id', $assignment_id)->get();

        $all_details = $ad_details->map(function ($ad_detail) {
            return [
                'detail_id' => Crypt::encrypt($ad_detail->id),
                'detail_title' => $ad_detail->assignment_detail_title,
                'detail_text' => $ad_detail->assignment_detail_text,
            ];
        });

        return response()->json($all_details);
    }

    public function deleteAssignmentDetails(Request $request)
    {
        $detail_id = Crypt::decrypt($request->detail);
        $assignment_detail = AssignmentDetails::find($detail_id);
        if ($assignment_detail->delete()) {
            return 1;
        } else {
            return 0;
        }
    }
}
