<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Admin\GuessPaperDetails;
use App\Models\Admin\GuessPaperFiles;
use App\Models\Admin\GuessPapers;
use App\Models\Admin\Template;
use App\Models\Admin\GuessPaperWhyChoose;
use App\Models\Admin\Coupons;
use App\Models\Admin\EducationalSessions;
use App\Models\Admin\Medium;
use App\Models\Admin\Subjects;
use App\Models\Admin\Types;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Yajra\DataTables\Facades\DataTables;

class GuessPapersController extends Controller
{
    public function checkGuessPaperLink($guess_paper_url, $encrypted_id = "")
    {
        $guess_paper_url = str_replace(['/', ' '], '-', $guess_paper_url);
        $guess_paper_url = preg_replace('/[^a-z0-9-]+/', '-', $guess_paper_url);
        $guess_paper_url = trim($guess_paper_url, '-');
        $guess_paper_url = preg_replace('/-+/', '-', $guess_paper_url);
        $original_link = $guess_paper_url;

        $id = (isset($encrypted_id) && !empty($encrypted_id)) ? Crypt::decrypt($encrypted_id) : 0;
        $suffix = 1;
        do {
            $count = $id != 0 ? GuessPapers::where('guess_paper_url', $guess_paper_url)->where('id', '!=', $id)->count() : GuessPapers::where('guess_paper_url', $guess_paper_url)->count();

            if ($count > 0) {
                $guess_paper_url = $original_link . '-' . $suffix;
                $suffix++;
            } else {
                break;
            }
        } while (true);

        return $guess_paper_url;
    }

    public function manageGuessPapers(Request $request)
    {
        $main_page = 'guess_papers_management';
        $currentPage = "manage_guess_papers";
        return view('admin.manage_guess_papers', ['courseId' => $request->course, 'main_page' => $main_page, 'currentPage' => $currentPage]);
    }

    public function manageGuessPapersData(Request $request)
    {
        $guess_papers = GuessPapers::with('subject')
            ->select(
                'id',
                'subject_id',
                'position_order',
                'guess_paper_code',
                'guess_paper_title',
                'guess_paper_image',
                'status'
            )->orderBy('position_order')->get();

        $model = Crypt::encrypt('GuessPapers');

        $data = [];
        $sno = 1;
        foreach ($guess_papers as $guess_paper) {
            $guess_paper->encrypted_id = Crypt::encrypt($guess_paper->id);
            $payload = compact('model', 'guess_paper', 'sno');
            $data[] = [
                'checkbox' => '<input type="checkbox" class="row_checkbox" value="' . $guess_paper->encrypted_id . '">',
                'id' => $sno++,
                'subject' => $guess_paper->subject->subject_code . ' (' . $guess_paper->subject->subject_name . ')',
                'guess_paper_code' => '<strong>' . $guess_paper->guess_paper_code . '</strong>',
                'guess_paper_title' => $guess_paper->guess_paper_title,
                'guess_paper_image' => '<ul class="list-unstyled users-list m-0 avatar-group d-flex align-items-center">
                    <li data-bs-toggle="tooltip" data-popup="tooltip-custom" data-bs-placement="top" class="avatar avatar-md pull-up" title="' . $guess_paper->guess_paper_title . '" >
                        <img src="' . asset($guess_paper->guess_paper_image) . '" class="rounded-circle" width="40" height="40">
                    </li>
                </ul>',
                'position_order' => $guess_paper->position_order,
                'status' => view('admin.components.status', $payload)->render(),
                'action' => view('admin.components.action', $payload)->render(),
            ];
        }
        return DataTables::of($data)->rawColumns(['checkbox', 'guess_paper_code', 'guess_paper_image', 'status', 'action'])->make(true);
    }

    public function bulkStatus(Request $request)
    {
        $request->validate([
            'encrypted_ids' => 'required|array',
            'encrypted_ids.*' => 'required|string',
        ]);

        try {
            $ids = collect($request->encrypted_ids)->map(function ($encrypted_id) {
                return Crypt::decrypt($encrypted_id);
            })->filter()->values();

            if ($ids->isEmpty()) {
                return response()->json([
                    'status'  => 'error',
                    'title'   => 'Invalid selection',
                    'message' => 'No valid guess papers selected.'
                ], 422);
            }

            $status_updated = GuessPapers::whereIn('id', $ids)
                ->update([
                    'status' => DB::raw("
                        CASE
                            WHEN status = 'active' THEN 'deactive'
                            WHEN status = 'deactive' THEN 'active'
                            ELSE status
                        END
                    ")
                ]);

            if ($status_updated > 0) {
                return response()->json(['status' => 'done', 'title' => "Status operation is done for selected guess papers.", 'message' => "Your files have been updated."], 200);
            }

            return response()->json([
                'status'  => 'error',
                'title' => "Failed to update!",
                'message' => "Your data is safe :)"
            ], 404);
        } catch (\Exception $e) {

            Log::error('Bulk update exception occurred', [
                'error'     => $e->getMessage(),
                'trace'     => $e->getTraceAsString(),
                'ip'        => $request->ip(),
            ]);

            return response()->json([
                'status'  => 'error',
                'title'   => 'Update Failed',
                'message' => 'Something went wrong while updating guess papers.'
            ], 500);
        }
    }

    public function bulkDelete(Request $request)
    {
        $request->validate([
            'encrypted_ids' => 'required|array',
            'encrypted_ids.*' => 'required|string',
        ]);

        try {
            $ids = collect($request->encrypted_ids)->map(function ($encrypted_id) {
                return Crypt::decrypt($encrypted_id);
            })->filter()->values();

            if ($ids->isEmpty()) {
                return response()->json([
                    'status'  => 'error',
                    'title'   => 'Invalid selection',
                    'message' => 'No valid guess papers selected.'
                ], 422);
            }

            $deleted = GuessPapers::whereIn('id', $ids)->delete();

            if ($deleted > 0) {
                return response()->json(['status' => 'done', 'title' => "Delete operation is done for selected guess papers.", 'message' => "Your files have been deleted."], 200);
            }

            return response()->json([
                'status'  => 'error',
                'title' => "Failed to Delete!",
                'message' => "Your data is safe :)"
            ], 404);
        } catch (\Exception $e) {

            Log::error('Bulk delete exception occurred', [
                'error'     => $e->getMessage(),
                'trace'     => $e->getTraceAsString(),
                'ip'        => $request->ip(),
            ]);

            return response()->json([
                'status'  => 'error',
                'title'   => 'Delete Failed',
                'message' => 'Something went wrong while deleting guess papers.'
            ], 500);
        }
    }

    private function decryptArray(?array $items): array
    {
        return collect($items ?? [])
            ->map(fn($id) => Crypt::decrypt($id))
            ->all();
    }

    private function safe(?string $text): ?string
    {
        return $text ? htmlspecialchars($text, ENT_QUOTES) : null;
    }

    private function uploadFile(Request $request, string $field, string $path): ?string
    {
        if (!$request->hasFile($field)) {
            return null;
        }

        return $this->storeImage($request->file($field), $path);
    }

    private function getDynamicDescription($subject_id, $languageIds, $sessionIds, $template_id, $code)
    {
        $subject = Subjects::findOrFail($subject_id);

        $languages = Medium::whereIn('id', $languageIds)
            ->pluck('language_name')
            ->implode(', ');

        $sessions = EducationalSessions::whereIn('id', $sessionIds)
            ->pluck('session_name')
            ->implode(', ');

        $template = Template::find($template_id);
        $short_description = $template->short_description;
        $download_info = $template->download_info;
        $description = $template->description;
        $cover_page_info = $template->cover_page_info;
        // $terms_conditions_text = $template->terms_conditions_text;

        $short_description = str_replace(
            [
                '{ CODE }',
                '{ SUBJECT_NAME }',
                '{ SESSIONS }',
                '{ LANGUAGES }',
            ],
            [
                $code,
                $subject->subject_name,
                $sessions,
                $languages,
            ],
            $short_description
        );

        $download_info = str_replace(
            [
                '{ CODE }',
                '{ SUBJECT_NAME }',
                '{ SESSIONS }',
                '{ LANGUAGES }',
            ],
            [
                $code,
                $subject->subject_name,
                $sessions,
                $languages,
            ],
            $download_info
        );

        $description = str_replace(
            [
                '{ CODE }',
                '{ SUBJECT_NAME }',
                '{ SESSIONS }',
                '{ LANGUAGES }',
            ],
            [
                $code,
                $subject->subject_name,
                $sessions,
                $languages,
            ],
            $description
        );

        $cover_page_info = str_replace(
            [
                '{ CODE }',
                '{ SUBJECT_NAME }',
                '{ SESSIONS }',
                '{ LANGUAGES }',
            ],
            [
                $code,
                $subject->subject_name,
                $sessions,
                $languages,
            ],
            $cover_page_info
        );

        return [$short_description, $download_info, $description, $cover_page_info];
    }

    public function manageGuessPaperTemplate(Request $request)
    {
        $guess_paper_template = Template::find(3);

        if (!$request->isMethod('post')) {
            return view('admin.template', [
                'guess_paper_template' => $guess_paper_template,
                'main_page' => 'guess_papers_management',
                'currentPage' => 'guess_paper_template',
            ]);
        }

        DB::beginTransaction();

        try {
            $guess_paper_template->update([
                'template_type' => 'guess_paper',
                'short_description' => $this->safe($request->short_description),
                'download_info' => $this->safe($request->download_info),
                'description' => $this->safe($request->description),
                'cover_page_info' => $this->safe($request->cover_page_info),
                // 'terms_conditions_text' => $this->safe($request->terms_conditions_text),
            ]);

            DB::commit();

            return redirect()
                ->route('manage.guess.paper.template')
                ->with('success', 'Guess Paper Template updated successfully!');
        } catch (\Throwable $e) {
            DB::rollBack();

            Log::error('Guess Paper template updation failed', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return redirect()
                ->route('manage.guess.paper.template')
                ->with('error', 'Something went wrong while updating guess paper template.');
        }
    }

    public function addGuessPaper(Request $request)
    {
        if (!$request->isMethod('post')) {
            return $this->renderGuessPaperForm($request);
        }

        $validated = $request->validate([
            'subject_id' => 'required|string',

            'type_ids' => 'array|nullable',
            'language_ids' => 'array|nullable',
            'session_ids' => 'array|nullable',
            'coupon_ids' => 'array|nullable',

            'guess_paper_code' => 'required|string',
            'guess_paper_url' => 'required|string',
            'guess_paper_title' => 'required|string',
            'maximum_retail_price' => 'required|numeric',
            'selling_price' => 'required|numeric',
        ]);

        DB::beginTransaction();

        try {
            $position_order = (GuessPapers::max('position_order') ?? 0) + 1;

            if (!empty($request->guess_paper_url)) {
                $link = strtolower($request->guess_paper_url);
            } else {
                $link = strtolower($request->guess_paper_code);
            }
            $guess_paper_url = $this->checkGuessPaperLink($link);
            $subject_id = Crypt::decrypt($request->subject_id);
            $languageIds = $this->decryptArray($request->language_ids);
            $sessionIds  = $this->decryptArray($request->session_ids);

            [$short_description, $download_info, $description, $cover_page_info] = $this->getDynamicDescription($subject_id, $languageIds, $sessionIds, 1, $request->guess_paper_code);

            $guess_paper = GuessPapers::create([
                'subject_id' => $subject_id,
                'position_order' => $position_order,

                'type_ids' => $this->decryptArray($request->type_ids),
                'language_ids' => $languageIds,
                'session_ids' => $sessionIds,
                'coupon_ids' => $this->decryptArray($request->coupon_ids),

                'guess_paper_code' => $request->guess_paper_code,
                'guess_paper_subject_name' => $request->guess_paper_subject_name,
                'guess_paper_url' => $guess_paper_url,
                'guess_paper_title' => $request->guess_paper_title,

                'breadcrumb_headline' => $request->breadcrumb_headline,
                'guess_paper_quantity' => $request->guess_paper_quantity,

                'maximum_retail_price' => $request->maximum_retail_price,
                'selling_price' => $request->selling_price,
                'discount' => $request->discount,
                'additional_discount' => $request->additional_discount,
                'discount_offer_expiry' => $request->discount_offer_expiry,
                'guess_paper_expiry' => $request->guess_paper_expiry,

                'short_description' => $short_description,
                'download_info' => $download_info,
                'description' => $description,
                'cover_page_info' => $cover_page_info,
                // 'terms_conditions_text' => $this->safe($request->terms_conditions_text),

                'meta_title' => $request->meta_title,
                'meta_keyword' => $request->meta_keyword,
                'meta_description' => $this->safe($request->meta_description),
            ]);

            // file uploads
            $guess_paper->update([
                'guess_paper_image' => $this->uploadFile($request, 'guess_paper_image', 'images/guess_papers/'),
                'question_paper_image' => $this->uploadFile($request, 'question_paper_image', 'images/guess_papers/question_paper_images/')
            ]);

            DB::commit();

            return redirect()
                ->route('manage.guess.papers', $request->course)
                ->with('success', 'Guess Paper inserted successfully!');
        } catch (\Throwable $e) {
            DB::rollBack();

            Log::error('Guess Paper insert failed', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return redirect()
                ->route('add.guess.paper', $request->course)
                ->with('error', 'Something went wrong while saving guess paper.');
        }
    }

    private function renderGuessPaperForm($request)
    {
        $all_guess_paper_types = Types::active()->get()->each(fn($i) => $i->encrypted_id = Crypt::encrypt($i->id));
        $all_guess_paper_languages = Medium::active()->get()->each(fn($i) => $i->encrypted_id = Crypt::encrypt($i->id));
        $all_guess_paper_sessions = EducationalSessions::active()->get()->each(fn($i) => $i->encrypted_id = Crypt::encrypt($i->id));
        $all_guess_paper_coupons = Coupons::where('reference_type', 'guess_paper')
            ->active()
            ->get()
            ->each(fn($i) => $i->encrypted_id = Crypt::encrypt($i->id));
        $all_subjects = Subjects::all();
        foreach ($all_subjects as $subject) {
            $subject->encrypted_id = Crypt::encrypt($subject->id);
        }

        return view('admin.guess_paper-ops', [
            'all_subjects' => $all_subjects,
            'all_guess_paper_types' => $all_guess_paper_types,
            'all_guess_paper_languages' => $all_guess_paper_languages,
            'all_guess_paper_sessions' => $all_guess_paper_sessions,
            'all_guess_paper_coupons' => $all_guess_paper_coupons,
            'main_page' => 'guess_papers_management',
            'currentPage' => 'manage_guess_papers',
        ]);
    }

    public function editGuessPaper(Request $request)
    {
        if ($request->isMethod('post')) {
            $request->validate([
                'subject_id' => 'required|string',
                'type_ids' => 'array|nullable',
                'language_ids' => 'array|nullable',
                'session_ids' => 'array|nullable',
                'coupon_ids' => 'array|nullable',
                'guess_paper_code' => 'required|string',
                'guess_paper_url' => 'required|string',
                'guess_paper_title' => 'required|string',
                'maximum_retail_price' => 'required|numeric',
                'selling_price' => 'required|numeric',
                'included_section_icons' => 'array|nullable',
                'included_section_titles' => 'array|nullable',
                'included_section_descriptions' => 'array|nullable',
            ], [
                'guess_paper_code.required' => 'Please provide a Guess Paper Code.',
                'guess_paper_code.string' => 'Guess Paper Code must be a string.',
                'guess_paper_title.required' => 'Please provide Guess Paper Title.',
                'guess_paper_title.string' => 'Guess Paper Title must be a string.',
                'maximum_retail_price.required' => 'Please provide MRP.',
                'maximum_retail_price.numeric' => 'MRP must be in numeric.',
                'selling_price.required' => 'Please provide Selling Price.',
                'selling_price.numeric' => 'Selling Price must be in numeric.',
            ]);

            $type_ids = array_map(function ($id) {
                return Crypt::decrypt($id);
            }, $request->type_ids ?? []);
            $language_ids = array_map(function ($id) {
                return Crypt::decrypt($id);
            }, $request->language_ids ?? []);
            $session_ids = array_map(function ($id) {
                return Crypt::decrypt($id);
            }, $request->session_ids ?? []);
            $coupon_ids = array_map(function ($id) {
                return Crypt::decrypt($id);
            }, $request->coupon_ids ?? []);

            $included_section_descriptions = array_map(function ($desc) {
                return htmlspecialchars($desc, ENT_QUOTES);
            }, $request->included_section_descriptions ?? []);

            if (!empty($request->guess_paper_url)) {
                $link = strtolower($request->guess_paper_url);
            } else {
                $link = strtolower($request->guess_paper_code);
            }
            $guess_paper_url = $this->checkGuessPaperLink($link, $request->guess_paper);

            $subject_id = Crypt::decrypt($request->subject_id);

            [$short_description, $download_info, $description, $cover_page_info] = $this->getDynamicDescription($subject_id, $language_ids, $session_ids, 1, $request->guess_paper_code);

            $id = Crypt::decrypt($request->guess_paper);
            $guess_paper = GuessPapers::findOrFail($id);
            $guess_paper->subject_id = $subject_id;
            $guess_paper->type_ids = $type_ids;
            $guess_paper->language_ids = $language_ids;
            $guess_paper->session_ids = $session_ids;
            $guess_paper->coupon_ids = $coupon_ids;
            $guess_paper->guess_paper_code = $request->guess_paper_code;
            $guess_paper->guess_paper_subject_name = $request->guess_paper_subject_name;
            $guess_paper->guess_paper_url = $guess_paper_url;
            $guess_paper->guess_paper_title = $request->guess_paper_title;
            $guess_paper->breadcrumb_headline = $request->breadcrumb_headline;
            $guess_paper->guess_paper_quantity = $request->guess_paper_quantity;
            $guess_paper->maximum_retail_price = $request->maximum_retail_price;
            $guess_paper->selling_price = $request->selling_price;
            $guess_paper->discount = $request->discount;
            $guess_paper->additional_discount = $request->additional_discount;
            $guess_paper->discount_offer_expiry = $request->discount_offer_expiry;
            $guess_paper->guess_paper_expiry = $request->guess_paper_expiry;
            $guess_paper->short_description = isset($request->short_description) ? htmlspecialchars($request->short_description, ENT_QUOTES) : $short_description;
            $guess_paper->download_info = isset($request->download_info) ? htmlspecialchars($request->download_info, ENT_QUOTES) : $download_info;
            $guess_paper->description = isset($request->description) ? htmlspecialchars($request->description, ENT_QUOTES) : $description;
            $guess_paper->cover_page_info = isset($request->cover_page_info) ? htmlspecialchars($request->cover_page_info, ENT_QUOTES) : $cover_page_info;
            // $guess_paper->terms_conditions_text = htmlspecialchars($request->terms_conditions_text, ENT_QUOTES);
            $guess_paper->meta_title = $request->meta_title;
            $guess_paper->meta_keyword = $request->meta_keyword;
            $guess_paper->meta_description = htmlspecialchars($request->meta_description, ENT_QUOTES);

            if (!empty($request->file('guess_paper_image'))) {
                $path = 'images/guess_papers/';
                $filePath = $this->storeImage($request->file("guess_paper_image"), $path, $guess_paper->guess_paper_image);
                $guess_paper->guess_paper_image = $filePath;
            }
            if (!empty($request->file('question_paper_image'))) {
                $path = 'images/guess_papers/question_paper_images/';
                $filePath = $this->storeImage($request->file("question_paper_image"), $path, $guess_paper->question_paper_image);
                $guess_paper->question_paper_image = $filePath;
            }

            if ($guess_paper->save()) {
                $request->session()->flash('success', 'Guess Paper is updated Successfully!');
                return redirect()->route('manage.guess.papers', $request->course);
            } else {
                $request->session()->flash('error', 'Updation Error!');
                return redirect()->route('edit.guess.paper', ["course" => $request->course, "guess_paper" => $request->guess_paper]);
            }
        } else {
            $all_guess_paper_types = Types::select('id', 'type_name')
                ->where('status', 'active')
                ->get()
                ->each(fn($item) => $item->encrypted_id = Crypt::encrypt($item->id));

            $all_guess_paper_languages = Medium::select('id', 'language_name')
                ->where('status', 'active')
                ->get()
                ->each(fn($item) => $item->encrypted_id = Crypt::encrypt($item->id));

            $all_guess_paper_sessions = EducationalSessions::select('id', 'session_name')
                ->where('status', 'active')
                ->get()
                ->each(fn($item) => $item->encrypted_id = Crypt::encrypt($item->id));

            $all_guess_paper_coupons = Coupons::select('id', 'coupon', 'coupon_name')
                ->where([
                    'reference_type' => 'guess_paper',
                    'status' => 'active'
                ])
                ->get()
                ->each(fn($item) => $item->encrypted_id = Crypt::encrypt($item->id));

            $all_subjects = Subjects::all();
            foreach ($all_subjects as $subject) {
                $subject->encrypted_id = Crypt::encrypt($subject->id);
            }

            $id = Crypt::decrypt($request->guess_paper);
            $guess_paper = GuessPapers::where('id', $id)->firstOrFail();
            $guess_paper->encrypted_id = $request->guess_paper;

            $included_section_icons = $guess_paper->included_icons;
            $included_section_titles = $guess_paper->included_titles;
            $included_section_descriptions = $guess_paper->included_descriptions;

            $main_page = 'guess_papers_management';
            $currentPage = "manage_guess_papers";
            return view('admin.guess_paper-ops', ['all_subjects' => $all_subjects, "all_guess_paper_types" => $all_guess_paper_types, "all_guess_paper_languages" => $all_guess_paper_languages, "all_guess_paper_sessions" => $all_guess_paper_sessions, "all_guess_paper_coupons" => $all_guess_paper_coupons, "guess_paper" => $guess_paper, 'main_page' => $main_page, 'currentPage' => $currentPage], compact('included_section_icons', 'included_section_titles', 'included_section_descriptions'));
        }
    }

    public function manageGuessPaperFiles(Request $request)
    {
        try {
            $guess_paper_id = Crypt::decrypt($request->guess_paper);
        } catch (\Exception $e) {
            return back()->with('error', 'Invalid guess paper reference.');
        }

        $guess_paper = GuessPapers::find($guess_paper_id);

        if (!$guess_paper) {
            return back()->with('error', 'Guess Paper not found.');
        }

        if ($request->isMethod('post')) {
            try {

                $request->validate([
                    'rule_ids'     => 'array|nullable',
                    'rule_ids.*'   => 'nullable|string',

                    'file_types'   => 'required|array',
                    'file_types.*' => 'required|string',

                    'file_languages'    => 'required|array',
                    'file_languages.*'  => 'required|string',

                    'files'        => 'array|nullable',
                    'files.*'      => 'nullable|file|max:10240', // 10MB

                    'file_passwords'   => 'array|nullable',
                    'file_passwords.*' => 'nullable|string|min:4',
                    // 'is_protected'     => 'array|nullable',
                ]);

                DB::beginTransaction();

                $existingMaxOrder = GuessPaperFiles::where('guess_paper_id', $guess_paper_id)
                    ->max('position_order');

                $position = $existingMaxOrder ? $existingMaxOrder + 1 : 1;

                foreach ($request->file_types as $i => $typeId) {

                    $langId   = $request->file_languages[$i] ?? null;
                    $ruleEnc  = $request->rule_ids[$i] ?? null;
                    $file     = $request->file('files')[$i] ?? null;

                    // $isProtected = $request->is_protected[$i] ?? 0;
                    $password    = $request->file_passwords[$i] ?? null;

                    // if ($isProtected && empty($password)) {
                    //     throw new \Exception("Password required for protected file");
                    // }
                    if (empty($password)) {
                        throw new \Exception("Password required for protected file");
                    }

                    // -----------------------------
                    // UPDATE EXISTING
                    // -----------------------------
                    if (!empty($ruleEnc)) {

                        $ruleId = Crypt::decrypt($ruleEnc);

                        $record = GuessPaperFiles::where('id', $ruleId)
                            ->where('guess_paper_id', $guess_paper_id)
                            ->first();

                        if (!$record) continue;

                        $data = [
                            'type_id'     => Crypt::decrypt($typeId),
                            'language_id' => Crypt::decrypt($langId),
                            // 'is_protected' => $isProtected,
                        ];

                        // if ($isProtected && $password) {
                        //     $data['file_password'] = Crypt::encrypt($password);
                        // }
                        if ($password) {
                            $data['file_password'] = $password;
                        }

                        // New file uploaded?
                        if ($file) {
                            $path = 'files/guess_papers/';
                            $filePath = $this->storeImage($file, $path);

                            $data['file_path'] = $path;
                            $data['file_name'] = $file->getClientOriginalName();
                            $data['file_full_path'] = $filePath;
                        }

                        $record->update($data);
                    }

                    // -----------------------------
                    // INSERT NEW
                    // -----------------------------
                    else {

                        if (!$file) {
                            throw new \Exception("File is required for new row");
                        }

                        $path = 'files/assignments/';
                        $filePath = $this->storeImage($file, $path);

                        GuessPaperFiles::create([
                            'guess_paper_id'     => $guess_paper_id,
                            'type_id'           => Crypt::decrypt($typeId),
                            'language_id'       => Crypt::decrypt($langId),
                            'position_order'    => $position++,
                            'file_path'         => $path,
                            'file_name'         => $file->getClientOriginalName(),
                            'file_full_path'    => $filePath,
                            // 'is_protected'      => $isProtected,
                            // 'file_password'     => $isProtected && $password ? Crypt::encrypt($password) : null,
                            'file_password'     => $password,
                        ]);
                    }
                }

                DB::commit();

                return redirect()
                    ->route('manage.guess.paper.files', $request->guess_paper)
                    ->with('success', 'Guess Paper files saved successfully!');
            } catch (\Throwable $e) {

                DB::rollBack();

                Log::error('Guess Paper files save failed', [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString(),
                ]);

                return redirect()
                    ->back()
                    ->withInput()
                    ->with('error', 'Something went wrong while saving files.');
            }
        }

        $all_guess_paper_types = Types::select('id', 'type_name')
            ->where('status', 'active')
            ->whereIn('id', $guess_paper->type_ids)
            ->get()
            ->each(fn($item) => $item->encrypted_id = Crypt::encrypt($item->id));

        $all_guess_paper_languages = Medium::select('id', 'language_name')
            ->where('status', 'active')
            ->whereIn('id', $guess_paper->language_ids)
            ->get()
            ->each(fn($item) => $item->encrypted_id = Crypt::encrypt($item->id));

        $main_page  = 'guess_papers_management';
        $currentPage = 'manage_guess_papers';

        return view('admin.guess_paper_files', [
            'guess_paperId' => $request->guess_paper,
            'guess_paper' => $guess_paper,
            'types' => $all_guess_paper_types,
            'languages' => $all_guess_paper_languages,
            'main_page'    => $main_page,
            'currentPage' => $currentPage,
        ]);
    }

    public function getGuessPaperFiles(Request $request)
    {
        $guess_paper_id = Crypt::decrypt($request->assignment);
        $adGuessPaperFiles = GuessPaperFiles::select('id', 'type_id', 'language_id', 'file_name', 'file_full_path', 'file_password', 'status')->where('guess_paper_id', $guess_paper_id)->get();

        $all_guessPaperFiles = $adGuessPaperFiles->map(function ($adGuessPaperFile) {
            return [
                'rule_id' => Crypt::encrypt($adGuessPaperFile->id),
                'file_type_id' => $adGuessPaperFile->type_id,
                'language_id' => $adGuessPaperFile->language_id,
                'file_name' => $adGuessPaperFile->file_name,
                'file_url' => asset($adGuessPaperFile->file_full_path),
                'file_password' => $adGuessPaperFile->file_password,
            ];
        });

        return response()->json($all_guessPaperFiles);
    }

    public function deleteGuessPaperFile(Request $request)
    {
        $rule_id = Crypt::decrypt($request->rule_id);
        $guess_paper_file = GuessPaperFiles::find($rule_id);

        if ($this->removeImage($guess_paper_file->file_full_path)) {
            if ($guess_paper_file->delete()) {
                return 1;
            } else {
                return 0;
            }
        } else {
            return 2;
        }
    }

    public function manageGuessPaperWhyChoose(Request $request)
    {
        try {
            $guess_paper_id = Crypt::decrypt($request->guess_paper);
        } catch (\Exception $e) {
            return back()->with('error', 'Invalid guess paper reference.');
        }

        $guess_paper = GuessPapers::find($guess_paper_id);

        if (!$guess_paper) {
            return back()->with('error', 'Guess Paper not found.');
        }

        if ($request->isMethod('post')) {
            try {
                $request->validate([
                    'choose_ids'        => 'array|nullable',
                    'choose_ids.*'      => 'nullable|string',
                    'choose_contents'   => 'array|nullable',
                    'choose_contents.*' => 'nullable|string',
                ]);

                DB::beginTransaction();

                // SAFE update
                $guess_paper->why_choose_headline = $request->why_choose_headline;
                $guess_paper->save();

                // Ensure at least one content exists
                $contents = collect($request->choose_contents)->filter();
                if ($contents->isEmpty() && empty($request->choose_ids)) {
                    DB::rollBack();
                    return redirect()
                        ->route('manage.guess.paper.why.choose.content', $request->guess_paper)
                        ->with('error', 'At least one item is required!');
                }

                $existingMaxOrder = GuessPaperWhyChoose::where('guess_paper_id', $guess_paper_id)
                    ->max('position_order');

                $position = $existingMaxOrder ? $existingMaxOrder + 1 : 1;

                foreach ($request->choose_contents ?? [] as $i => $content) {
                    $content = trim($content);

                    if ($content === '') {
                        continue;
                    }

                    // UPDATE EXISTING
                    if (!empty($request->choose_ids[$i])) {
                        $whyChooseId = Crypt::decrypt($request->choose_ids[$i]);
                        $whyChoose   = GuessPaperWhyChoose::where('id', $whyChooseId)
                            ->where('guess_paper_id', $guess_paper_id)
                            ->first();

                        if ($whyChoose) {
                            $whyChoose->update([
                                'guess_paper_content' => $content,
                            ]);
                        }
                    }
                    // INSERT NEW
                    else {
                        GuessPaperWhyChoose::create([
                            'guess_paper_id'      => $guess_paper_id,
                            'position_order'     => $position++,
                            'guess_paper_content' => $content,
                        ]);
                    }
                }

                DB::commit();

                return redirect()
                    ->route('manage.guess.paper.why.choose.content', $request->guess_paper)
                    ->with('success', 'Guess Paper Why Choose content saved successfully!');
            } catch (\Throwable $e) {
                DB::rollBack();

                Log::error('Guess Paper Why Choose save failed', [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString(),
                ]);

                return redirect()
                    ->back()
                    ->withInput()
                    ->with('error', 'Something went wrong while saving data.');
            }
        }

        $main_page  = 'guess_papers_management';
        $currentPage = 'manage_guess_papers';

        return view('admin.guess_paper_why_choose_content', [
            'guessPaperId' => $request->guess_paper,
            'whyChooseHeadline' => $guess_paper->why_choose_headline,
            'main_page'    => $main_page,
            'currentPage' => $currentPage,
        ]);
    }

    public function getGuessPaperWhyChoose(Request $request)
    {
        $guess_paper_id = Crypt::decrypt($request->guess_paper);
        $adwhychooses = GuessPaperWhyChoose::select('id', 'guess_paper_content', 'status')->where('guess_paper_id', $guess_paper_id)->get();

        $all_whychooses = $adwhychooses->map(function ($adwhychoose) {
            return [
                'choose_id' => Crypt::encrypt($adwhychoose->id),
                'choose_content' => $adwhychoose->guess_paper_content,
            ];
        });

        return response()->json($all_whychooses);
    }

    public function deleteWhyChoose(Request $request)
    {
        $choose_id = Crypt::decrypt($request->choose);
        $guess_paper_why_choose = GuessPaperWhyChoose::find($choose_id);
        if ($guess_paper_why_choose->delete()) {
            return 1;
        } else {
            return 0;
        }
    }

    public function manageGuessPaperDetails(Request $request)
    {
        if ($request->isMethod('post')) {
            try {
                $request->validate([
                    'detail_ids'         => 'array|nullable',
                    'detail_ids.*'       => 'nullable|string',
                    'detail_titles'      => 'array|nullable',
                    'detail_titles.*'    => 'nullable|string',
                    'detail_texts'       => 'array|nullable',
                    'detail_texts.*'     => 'nullable|string',
                ]);

                DB::beginTransaction();

                // SAFE decrypt
                try {
                    $guess_paper_id = Crypt::decrypt($request->guess_paper);
                } catch (\Exception $e) {
                    return back()->with('error', 'Invalid guess paper reference.');
                }

                // Ensure at least one detail exists
                $titles = collect($request->detail_titles)->filter(fn($v) => trim($v) !== '');
                if ($titles->isEmpty()) {
                    DB::rollBack();
                    return redirect()
                        ->route('manage.guess.paper.details.contents', $request->guess_paper)
                        ->with('error', 'At least one detail title is required!');
                }

                // Get current max position
                $existingMaxOrder = GuessPaperDetails::where('guess_paper_id', $guess_paper_id)
                    ->max('position_order');

                $position = $existingMaxOrder ? $existingMaxOrder + 1 : 1;

                foreach ($request->detail_titles ?? [] as $i => $title) {
                    $title = trim($title);
                    $text  = trim($request->detail_texts[$i] ?? '');

                    if ($title === '') {
                        continue;
                    }

                    $detailId = $request->detail_ids[$i] ?? null;

                    // UPDATE EXISTING
                    if (!empty($detailId)) {
                        try {
                            $decodedId = Crypt::decrypt($detailId);
                        } catch (\Exception $e) {
                            continue;
                        }

                        $detail = GuessPaperDetails::where('id', $decodedId)
                            ->where('guess_paper_id', $guess_paper_id)
                            ->first();

                        if ($detail) {
                            $detail->update([
                                'guess_paper_detail_title' => $title,
                                'guess_paper_detail_text'  => $text,
                            ]);
                        }
                    }
                    // INSERT NEW
                    else {
                        GuessPaperDetails::create([
                            'guess_paper_id'            => $guess_paper_id,
                            'position_order'           => $position++,
                            'guess_paper_detail_title'  => $title,
                            'guess_paper_detail_text'   => $text,
                        ]);
                    }
                }

                DB::commit();

                return redirect()
                    ->route('manage.guess.paper.details.contents', $request->guess_paper)
                    ->with('success', 'Guess Paper details saved successfully!');
            } catch (\Throwable $e) {
                DB::rollBack();

                Log::error('Guess Paper Details save failed', [
                    'error' => $e->getMessage(),
                    'file'  => $e->getFile(),
                    'line'  => $e->getLine(),
                ]);

                return back()
                    ->withInput()
                    ->with('error', 'Something went wrong while saving details.');
            }
        }

        return view('admin.guess_paper_other_details', [
            'guessPaperId' => $request->guess_paper,
            'main_page'    => 'guess_papers_management',
            'currentPage' => 'manage_guess_papers',
        ]);
    }

    public function getGuessPaperDetails(Request $request)
    {
        $guess_paper_id = Crypt::decrypt($request->guess_paper);
        $ad_details = GuessPaperDetails::select('id', 'guess_paper_detail_title', 'guess_paper_detail_text', 'status')->where('guess_paper_id', $guess_paper_id)->get();

        $all_details = $ad_details->map(function ($ad_detail) {
            return [
                'detail_id' => Crypt::encrypt($ad_detail->id),
                'detail_title' => $ad_detail->guess_paper_detail_title,
                'detail_text' => $ad_detail->guess_paper_detail_text,
            ];
        });

        return response()->json($all_details);
    }

    public function deleteGuessPaperDetails(Request $request)
    {
        $detail_id = Crypt::decrypt($request->detail);
        $guess_paper_detail = GuessPaperDetails::find($detail_id);
        if ($guess_paper_detail->delete()) {
            return 1;
        } else {
            return 0;
        }
    }
}
