<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Admin\Coupons;
use App\Models\Admin\EducationalSessions;
use App\Models\Admin\Medium;
use App\Models\Admin\ProjectDetails;
use App\Models\Admin\Projects;
use App\Models\Admin\ProjectWhyChoose;
use App\Models\Admin\Subjects;
use App\Models\Admin\Template;
use App\Models\Admin\Types;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Yajra\DataTables\Facades\DataTables;

class ProjectsController extends Controller
{
    public function checkProjectLink($project_url, $encrypted_id = "")
    {
        $project_url = str_replace(['/', ' '], '-', $project_url);
        $project_url = preg_replace('/[^a-z0-9-]+/', '-', $project_url);
        $project_url = trim($project_url, '-');
        $project_url = preg_replace('/-+/', '-', $project_url);
        $original_link = $project_url;

        $id = (isset($encrypted_id) && !empty($encrypted_id)) ? Crypt::decrypt($encrypted_id) : 0;
        $suffix = 1;
        do {
            $count = $id != 0 ? Projects::where('project_url', $project_url)->where('id', '!=', $id)->count() : Projects::where('project_url', $project_url)->count();

            if ($count > 0) {
                $project_url = $original_link . '-' . $suffix;
                $suffix++;
            } else {
                break;
            }
        } while (true);

        return $project_url;
    }

    public function manageProjects(Request $request)
    {
        $main_page = 'projects_management';
        $currentPage = "manage_projects";
        return view('admin.manage_projects', ['courseId' => $request->course, 'main_page' => $main_page, 'currentPage' => $currentPage]);
    }

    public function manageProjectsData(Request $request)
    {
        $projects = Projects::with('subject')
            ->select(
                'id',
                'subject_id',
                'position_order',
                'project_code',
                'project_title',
                'project_image',
                'status'
            )->orderBy('position_order')->get();

        $model = Crypt::encrypt('Projects');

        $data = [];
        $sno = 1;
        foreach ($projects as $project) {
            $project->encrypted_id = Crypt::encrypt($project->id);
            $payload = compact('model', 'project', 'sno');
            $data[] = [
                'checkbox' => '<input type="checkbox" class="row_checkbox" value="' . $project->encrypted_id . '">',
                'id' => $sno++,
                'subject' => $project->subject->subject_code . ' (' . $project->subject->subject_name . ')',
                'project_code' => '<strong>' . $project->project_code . '</strong>',
                'project_title' => $project->project_title,
                'project_image' => '<ul class="list-unstyled users-list m-0 avatar-group d-flex align-items-center">
                    <li data-bs-toggle="tooltip" data-popup="tooltip-custom" data-bs-placement="top" class="avatar avatar-md pull-up" title="' . $project->project_title . '" >
                        <img src="' . asset($project->project_image) . '" class="rounded-circle" width="40" height="40">
                    </li>
                </ul>',
                'position_order' => $project->position_order,
                'status' => view('admin.components.status', $payload)->render(),
                'action' => view('admin.components.action', $payload)->render(),
            ];
        }
        return DataTables::of($data)->rawColumns(['checkbox', 'project_code', 'project_image', 'status', 'action'])->make(true);
    }

    public function bulkStatus(Request $request)
    {
        $request->validate([
            'encrypted_ids' => 'required|array',
            'encrypted_ids.*' => 'required|string',
        ]);

        try {
            $ids = collect($request->encrypted_ids)->map(function ($encrypted_id) {
                return Crypt::decrypt($encrypted_id);
            })->filter()->values();

            if ($ids->isEmpty()) {
                return response()->json([
                    'status'  => 'error',
                    'title'   => 'Invalid selection',
                    'message' => 'No valid projects selected.'
                ], 422);
            }

            $status_updated = Projects::whereIn('id', $ids)
                ->update([
                    'status' => DB::raw("
                        CASE
                            WHEN status = 'active' THEN 'deactive'
                            WHEN status = 'deactive' THEN 'active'
                            ELSE status
                        END
                    ")
                ]);

            if ($status_updated > 0) {
                return response()->json(['status' => 'done', 'title' => "Status operation is done for selected Projects.", 'message' => "Your files have been updated."], 200);
            }

            return response()->json([
                'status'  => 'error',
                'title' => "Failed to update!",
                'message' => "Your data is safe :)"
            ], 404);
        } catch (\Exception $e) {

            Log::error('Bulk update exception occurred', [
                'error'     => $e->getMessage(),
                'trace'     => $e->getTraceAsString(),
                'ip'        => $request->ip(),
            ]);

            return response()->json([
                'status'  => 'error',
                'title'   => 'Update Failed',
                'message' => 'Something went wrong while updating projects.'
            ], 500);
        }
    }

    public function bulkDelete(Request $request)
    {
        $request->validate([
            'encrypted_ids' => 'required|array',
            'encrypted_ids.*' => 'required|string',
        ]);

        try {
            $ids = collect($request->encrypted_ids)->map(function ($encrypted_id) {
                return Crypt::decrypt($encrypted_id);
            })->filter()->values();

            if ($ids->isEmpty()) {
                return response()->json([
                    'status'  => 'error',
                    'title'   => 'Invalid selection',
                    'message' => 'No valid projects selected.'
                ], 422);
            }

            $deleted = Projects::whereIn('id', $ids)->delete();

            if ($deleted > 0) {
                return response()->json(['status' => 'done', 'title' => "Delete operation is done for selected Projects.", 'message' => "Your files have been deleted."], 200);
            }

            return response()->json([
                'status'  => 'error',
                'title' => "Failed to Delete!",
                'message' => "Your data is safe :)"
            ], 404);
        } catch (\Exception $e) {

            Log::error('Bulk delete exception occurred', [
                'error'     => $e->getMessage(),
                'trace'     => $e->getTraceAsString(),
                'ip'        => $request->ip(),
            ]);

            return response()->json([
                'status'  => 'error',
                'title'   => 'Delete Failed',
                'message' => 'Something went wrong while deleting projects.'
            ], 500);
        }
    }

    private function decryptArray(?array $items): array
    {
        return collect($items ?? [])
            ->map(fn($id) => Crypt::decrypt($id))
            ->all();
    }

    private function safe(?string $text): ?string
    {
        return $text ? htmlspecialchars($text, ENT_QUOTES) : null;
    }

    private function uploadFile(Request $request, string $field, string $path): ?string
    {
        if (!$request->hasFile($field)) {
            return null;
        }

        return $this->storeImage($request->file($field), $path);
    }

    private function getDynamicDescription($subject_id, $languageIds, $sessionIds, $template_id, $code)
    {
        $subject = Subjects::findOrFail($subject_id);

        $languages = Medium::whereIn('id', $languageIds)
            ->pluck('language_name')
            ->implode(', ');

        $sessions = EducationalSessions::whereIn('id', $sessionIds)
            ->pluck('session_name')
            ->implode(', ');

        $template = Template::find($template_id);
        $short_description = $template->short_description;
        $download_info = $template->download_info;
        $description = $template->description;
        $cover_page_info = $template->cover_page_info;
        // $terms_conditions_text = $template->terms_conditions_text;

        $short_description = str_replace(
            [
                '{ CODE }',
                '{ SUBJECT_NAME }',
                '{ SESSIONS }',
                '{ LANGUAGES }',
            ],
            [
                $code,
                $subject->subject_name,
                $sessions,
                $languages,
            ],
            $short_description
        );

        $download_info = str_replace(
            [
                '{ CODE }',
                '{ SUBJECT_NAME }',
                '{ SESSIONS }',
                '{ LANGUAGES }',
            ],
            [
                $code,
                $subject->subject_name,
                $sessions,
                $languages,
            ],
            $download_info
        );

        $description = str_replace(
            [
                '{ CODE }',
                '{ SUBJECT_NAME }',
                '{ SESSIONS }',
                '{ LANGUAGES }',
            ],
            [
                $code,
                $subject->subject_name,
                $sessions,
                $languages,
            ],
            $description
        );

        $cover_page_info = str_replace(
            [
                '{ CODE }',
                '{ SUBJECT_NAME }',
                '{ SESSIONS }',
                '{ LANGUAGES }',
            ],
            [
                $code,
                $subject->subject_name,
                $sessions,
                $languages,
            ],
            $cover_page_info
        );

        return [$short_description, $download_info, $description, $cover_page_info];
    }

    public function manageProjectTemplate(Request $request)
    {
        $project_template = Template::find(2);

        if (!$request->isMethod('post')) {
            return view('admin.template', [
                'project_template' => $project_template,
                'main_page' => 'projects_management',
                'currentPage' => 'project_template',
            ]);
        }

        DB::beginTransaction();

        try {
            $project_template->update([
                'template_type' => 'project',
                'short_description' => $this->safe($request->short_description),
                'download_info' => $this->safe($request->download_info),
                'description' => $this->safe($request->description),
                'cover_page_info' => $this->safe($request->cover_page_info),
                // 'terms_conditions_text' => $this->safe($request->terms_conditions_text),
            ]);

            DB::commit();

            return redirect()
                ->route('manage.project.template')
                ->with('success', 'Project Template updated successfully!');
        } catch (\Throwable $e) {
            DB::rollBack();

            Log::error('Project template updation failed', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return redirect()
                ->route('manage.project.template')
                ->with('error', 'Something went wrong while updating project template.');
        }
    }

    public function addProject(Request $request)
    {
        if (!$request->isMethod('post')) {
            return $this->renderProjectForm($request);
        }

        $validated = $request->validate([
            'subject_id' => 'required|string',

            'type_ids' => 'array|nullable',
            'language_ids' => 'array|nullable',
            'session_ids' => 'array|nullable',
            'coupon_ids' => 'array|nullable',

            'project_code' => 'required|string',
            'project_url' => 'required|string',
            'project_title' => 'required|string',
            'maximum_retail_price' => 'required|numeric',
            'selling_price' => 'required|numeric',

            'project_file' => 'required',
            'file_password' => 'required',
        ]);

        DB::beginTransaction();

        try {
            $position_order = (Projects::max('position_order') ?? 0) + 1;
            $reference_number = (Projects::max('reference_number') ?? 0) + 100;

            if (!empty($request->project_url)) {
                $link = strtolower($request->project_url);
            } else {
                $link = strtolower($request->project_code);
            }
            $project_url = $this->checkProjectLink($link);
            $subject_id = Crypt::decrypt($request->subject_id);
            $languageIds = $this->decryptArray($request->language_ids);
            $sessionIds  = $this->decryptArray($request->session_ids);

            [$short_description, $download_info, $description, $cover_page_info] = $this->getDynamicDescription($subject_id, $languageIds, $sessionIds, 2, $request->project_code);

            $topics = explode(',', str_replace(', ', ',', $request->topics));

            $project = Projects::create([
                'subject_id' => Crypt::decrypt($request->subject_id),
                'position_order' => $position_order,

                'type_ids' => $this->decryptArray($request->type_ids),
                'language_ids' => $languageIds,
                'session_ids' => $sessionIds,
                'coupon_ids' => $this->decryptArray($request->coupon_ids),

                'topics' => $topics,
                'reference_number' => $reference_number,
                'project_code' => $request->project_code,
                'project_title' => $request->project_title,
                'project_subject_name' => $request->project_subject_name,
                'project_url' => $project_url,

                'breadcrumb_headline' => $request->breadcrumb_headline,
                'project_quantity' => $request->project_quantity,

                'maximum_retail_price' => $request->maximum_retail_price,
                'selling_price' => $request->selling_price,
                'discount' => $request->discount,
                'additional_discount' => $request->additional_discount,
                'discount_offer_expiry' => $request->discount_offer_expiry,
                'project_expiry' => $request->project_expiry,

                'short_description' => $short_description,
                'download_info' => $download_info,
                'description' => $description,
                'cover_page_info' => $cover_page_info,

                'meta_title' => $request->meta_title,
                'meta_keyword' => $request->meta_keyword,
                'meta_description' => $this->safe($request->meta_description),
            ]);

            // file uploads
            $project->update([
                'project_image' => $this->uploadFile($request, 'project_image', 'images/projects/'),
                // 'question_paper_image' => $this->uploadFile($request, 'question_paper_image', 'images/projects/question_paper_images/'),
                'demo_file' => $this->uploadFile($request, 'demo_file', 'images/projects/demo/'),
                'project_file' => $this->uploadFile($request, 'project_file', 'files/projects/'),
                'file_password' => $request->file_password,
            ]);

            DB::commit();

            return redirect()
                ->route('manage.projects')
                ->with('success', 'Project inserted successfully!');
        } catch (\Throwable $e) {
            DB::rollBack();

            Log::error('Project insert failed', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return redirect()
                ->route('add.project')
                ->with('error', 'Something went wrong while saving project.');
        }
    }

    private function renderProjectForm($request)
    {
        $all_project_types = Types::active()->get()->each(fn($i) => $i->encrypted_id = Crypt::encrypt($i->id));
        $all_project_languages = Medium::active()->get()->each(fn($i) => $i->encrypted_id = Crypt::encrypt($i->id));
        // $all_project_sessions = EducationalSessions::active()->get()->each(fn($i) => $i->encrypted_id = Crypt::encrypt($i->id));
        $all_project_coupons = Coupons::where('reference_type', 'project')
            ->get()
            ->each(fn($i) => $i->encrypted_id = Crypt::encrypt($i->id));
        $all_subjects = Subjects::all();
        foreach ($all_subjects as $subject) {
            $subject->encrypted_id = Crypt::encrypt($subject->id);
        }

        return view('admin.project-ops', [
            'all_subjects' => $all_subjects,
            'all_project_types' => $all_project_types,
            'all_project_languages' => $all_project_languages,
            // 'all_project_sessions' => $all_project_sessions,
            'all_project_coupons' => $all_project_coupons,
            'main_page' => 'projects_management',
            'currentPage' => 'manage_projects',
        ]);
    }

    public function editProject(Request $request)
    {
        if ($request->isMethod('post')) {
            $request->validate([
                'subject_id' => 'required|string',
                'type_ids' => 'array|nullable',
                'language_ids' => 'array|nullable',
                'session_ids' => 'array|nullable',
                'coupon_ids' => 'array|nullable',
                'project_code' => 'required|string',
                'project_title' => 'required|string',
                'project_url' => 'required|string',
                'maximum_retail_price' => 'required|numeric',
                'selling_price' => 'required|numeric',
                'included_section_icons' => 'array|nullable',
                'included_section_titles' => 'array|nullable',
                'included_section_descriptions' => 'array|nullable',
                // 'project_file' => 'required',
                'file_password' => 'required',
            ], [
                'project_code.required' => 'Please provide a Project Code.',
                'project_code.string' => 'Project Code must be a string.',
                'project_title.required' => 'Please provide Project Title.',
                'project_title.string' => 'Project Title must be a string.',
                'maximum_retail_price.required' => 'Please provide MRP.',
                'maximum_retail_price.numeric' => 'MRP must be in numeric.',
                'selling_price.required' => 'Please provide Selling Price.',
                'selling_price.numeric' => 'Selling Price must be in numeric.',
                // 'project_file.required' => 'Please provide Project File.',
                'file_password.required' => 'Please provide File Password.',
            ]);

            $type_ids = array_map(function ($id) {
                return Crypt::decrypt($id);
            }, $request->type_ids ?? []);
            $language_ids = array_map(function ($id) {
                return Crypt::decrypt($id);
            }, $request->language_ids ?? []);
            $session_ids = array_map(function ($id) {
                return Crypt::decrypt($id);
            }, $request->session_ids ?? []);
            $coupon_ids = array_map(function ($id) {
                return Crypt::decrypt($id);
            }, $request->coupon_ids ?? []);

            $included_section_descriptions = array_map(function ($desc) {
                return htmlspecialchars($desc, ENT_QUOTES);
            }, $request->included_section_descriptions ?? []);

            if (!empty($request->project_url)) {
                $link = strtolower($request->project_url);
            } else {
                $link = strtolower($request->project_code);
            }
            $project_url = $this->checkProjectLink($link, $request->project);

            $subject_id = Crypt::decrypt($request->subject_id);

            [$short_description, $download_info, $description, $cover_page_info] = $this->getDynamicDescription($subject_id, $language_ids, $session_ids, 2, $request->project_code);

            $topics = explode(',', str_replace(', ', ',', $request->topics));

            $id = Crypt::decrypt($request->project);
            $project = Projects::findOrFail($id);
            $project->subject_id = $subject_id;
            $project->type_ids = $type_ids;
            $project->language_ids = $language_ids;
            $project->session_ids = $session_ids;
            $project->coupon_ids = $coupon_ids;
            $project->topics = $topics;
            // $project->reference_number = $request->reference_number ?? rand(100, 9999);
            $project->project_code = $request->project_code;
            $project->project_title = $request->project_title;
            $project->project_subject_name = $request->project_subject_name;
            $project->project_url = $project_url;
            $project->breadcrumb_headline = $request->breadcrumb_headline;
            $project->project_quantity = $request->project_quantity;
            $project->maximum_retail_price = $request->maximum_retail_price;
            $project->selling_price = $request->selling_price;
            $project->discount = $request->discount;
            $project->additional_discount = $request->additional_discount;
            $project->discount_offer_expiry = $request->discount_offer_expiry;
            $project->project_expiry = $request->project_expiry;
            $project->short_description = isset($request->short_description) ? htmlspecialchars($request->short_description, ENT_QUOTES) : $short_description;
            $project->download_info = isset($request->download_info) ? htmlspecialchars($request->download_info, ENT_QUOTES) : $download_info;
            $project->description = isset($request->description) ? htmlspecialchars($request->description, ENT_QUOTES) : $description;
            $project->cover_page_info = isset($request->cover_page_info) ? htmlspecialchars($request->cover_page_info, ENT_QUOTES) : $cover_page_info;
            $project->file_password = $request->file_password;
            $project->meta_title = $request->meta_title;
            $project->meta_keyword = $request->meta_keyword;
            $project->meta_description = htmlspecialchars($request->meta_description, ENT_QUOTES);

            if (!empty($request->file('project_image'))) {
                $path = 'images/projects/';
                $filePath = $this->storeImage($request->file("project_image"), $path, $project->project_image);
                $project->project_image = $filePath;
            }
            // if (!empty($request->file('question_paper_image'))) {
            //     $path = 'images/projects/question_paper_images/';
            //     $filePath = $this->storeImage($request->file("question_paper_image"), $path, $project->question_paper_image);
            //     $project->question_paper_image = $filePath;
            // }
            if (!empty($request->file('demo_file'))) {
                $path = 'images/projects/demo/';
                $filePath = $this->storeImage($request->file("demo_file"), $path, $project->demo_file);
                $project->demo_file = $filePath;
            }
            if (!empty($request->file('project_file'))) {
                $path = 'files/projects/';
                $filePath = $this->storeImage($request->file("project_file"), $path, $project->project_file);
                $project->project_file = $filePath;
            }

            if ($project->save()) {
                $request->session()->flash('success', 'Project is updated Successfully!');
                return redirect()->route('manage.projects');
            } else {
                $request->session()->flash('error', 'Updation Error!');
                return redirect()->route('edit.project', ["project" => $request->project]);
            }
        } else {
            $all_project_types = Types::select('id', 'type_name')
                ->where('status', 'active')
                ->get()
                ->each(fn($item) => $item->encrypted_id = Crypt::encrypt($item->id));

            $all_project_languages = Medium::select('id', 'language_name')
                ->where('status', 'active')
                ->get()
                ->each(fn($item) => $item->encrypted_id = Crypt::encrypt($item->id));

            // $all_project_sessions = EducationalSessions::select('id', 'session_name')
            //     ->where('status', 'active')
            //     ->get()
            //     ->each(fn($item) => $item->encrypted_id = Crypt::encrypt($item->id));

            $all_project_coupons = Coupons::select('id', 'coupon', 'coupon_name')
                ->where([
                    'reference_type' => 'project',
                    'status' => 'active'
                ])
                ->get()
                ->each(fn($item) => $item->encrypted_id = Crypt::encrypt($item->id));

            $all_subjects = Subjects::all();
            foreach ($all_subjects as $subject) {
                $subject->encrypted_id = Crypt::encrypt($subject->id);
            }

            $id = Crypt::decrypt($request->project);
            $project = Projects::where('id', $id)->firstOrFail();
            $project->encrypted_id = $request->project;

            $included_section_icons = $project->included_icons;
            $included_section_titles = $project->included_titles;
            $included_section_descriptions = $project->included_descriptions;

            // , "all_project_sessions" => $all_project_sessions

            $main_page = 'projects_management';
            $currentPage = "manage_projects";
            return view('admin.project-ops', ['all_subjects' => $all_subjects, "all_project_types" => $all_project_types, "all_project_languages" => $all_project_languages, "all_project_coupons" => $all_project_coupons, "project" => $project, 'main_page' => $main_page, 'currentPage' => $currentPage], compact('included_section_icons', 'included_section_titles', 'included_section_descriptions'));
        }
    }

    public function manageProjectWhyChoose(Request $request)
    {
        try {
            $project_id = Crypt::decrypt($request->project);
        } catch (\Exception $e) {
            return back()->with('error', 'Invalid Project reference.');
        }

        $project = Projects::find($project_id);

        if (!$project) {
            return back()->with('error', 'Project not found.');
        }

        if ($request->isMethod('post')) {
            try {
                $request->validate([
                    'choose_ids'        => 'array|nullable',
                    'choose_ids.*'      => 'nullable|string',
                    'choose_contents'   => 'array|nullable',
                    'choose_contents.*' => 'nullable|string',
                ]);

                DB::beginTransaction();

                // SAFE update
                $project->why_choose_headline = $request->why_choose_headline;
                $project->save();

                // Ensure at least one content exists
                $contents = collect($request->choose_contents)->filter();
                if ($contents->isEmpty() && empty($request->choose_ids)) {
                    DB::rollBack();
                    return redirect()
                        ->route('manage.project.why.choose.content', $request->project)
                        ->with('error', 'At least one item is required!');
                }

                $existingMaxOrder = ProjectWhyChoose::where('project_id', $project_id)
                    ->max('position_order');

                $position = $existingMaxOrder ? $existingMaxOrder + 1 : 1;

                foreach ($request->choose_contents ?? [] as $i => $content) {
                    $content = trim($content);

                    if ($content === '') {
                        continue;
                    }

                    // UPDATE EXISTING
                    if (!empty($request->choose_ids[$i])) {
                        $whyChooseId = Crypt::decrypt($request->choose_ids[$i]);
                        $whyChoose   = ProjectWhyChoose::where('id', $whyChooseId)
                            ->where('project_id', $project_id)
                            ->first();

                        if ($whyChoose) {
                            $whyChoose->update([
                                'project_content' => $content,
                            ]);
                        }
                    }
                    // INSERT NEW
                    else {
                        ProjectWhyChoose::create([
                            'project_id'      => $project_id,
                            'position_order'     => $position++,
                            'project_content' => $content,
                        ]);
                    }
                }

                DB::commit();

                return redirect()
                    ->route('manage.project.why.choose.content', $request->project)
                    ->with('success', 'Why Choose content saved successfully!');
            } catch (\Throwable $e) {
                DB::rollBack();

                Log::error('Why Choose save failed', [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString(),
                ]);

                return redirect()
                    ->back()
                    ->withInput()
                    ->with('error', 'Something went wrong while saving data.');
            }
        }

        $main_page  = 'projects_management';
        $currentPage = 'manage_projects';

        return view('admin.project_why_choose_content', [
            'projectId' => $request->project,
            'whyChooseHeadline' => $project->why_choose_headline,
            'main_page'    => $main_page,
            'currentPage' => $currentPage,
        ]);
    }

    public function getProjectWhyChoose(Request $request)
    {
        $project_id = Crypt::decrypt($request->project);
        $adwhychooses = ProjectWhyChoose::select('id', 'project_content', 'status')->where('project_id', $project_id)->get();

        $all_whychooses = $adwhychooses->map(function ($adwhychoose) {
            return [
                'choose_id' => Crypt::encrypt($adwhychoose->id),
                'choose_content' => $adwhychoose->project_content,
            ];
        });

        return response()->json($all_whychooses);
    }

    public function deleteWhyChoose(Request $request)
    {
        $choose_id = Crypt::decrypt($request->choose);
        $project_why_choose = ProjectWhyChoose::find($choose_id);
        if ($project_why_choose->delete()) {
            return 1;
        } else {
            return 0;
        }
    }

    public function manageProjectDetails(Request $request)
    {
        if ($request->isMethod('post')) {
            try {
                $request->validate([
                    'detail_ids'         => 'array|nullable',
                    'detail_ids.*'       => 'nullable|string',
                    'detail_titles'      => 'array|nullable',
                    'detail_titles.*'    => 'nullable|string',
                    'detail_texts'       => 'array|nullable',
                    'detail_texts.*'     => 'nullable|string',
                ]);

                DB::beginTransaction();

                // SAFE decrypt
                try {
                    $project_id = Crypt::decrypt($request->project);
                } catch (\Exception $e) {
                    return back()->with('error', 'Invalid Project reference.');
                }

                // Ensure at least one detail exists
                $titles = collect($request->detail_titles)->filter(fn($v) => trim($v) !== '');
                if ($titles->isEmpty()) {
                    DB::rollBack();
                    return redirect()
                        ->route('manage.project.details.contents', $request->project)
                        ->with('error', 'At least one detail title is required!');
                }

                // Get current max position
                $existingMaxOrder = ProjectDetails::where('project_id', $project_id)
                    ->max('position_order');

                $position = $existingMaxOrder ? $existingMaxOrder + 1 : 1;

                foreach ($request->detail_titles ?? [] as $i => $title) {
                    $title = trim($title);
                    $text  = trim($request->detail_texts[$i] ?? '');

                    if ($title === '') {
                        continue;
                    }

                    $detailId = $request->detail_ids[$i] ?? null;

                    // UPDATE EXISTING
                    if (!empty($detailId)) {
                        try {
                            $decodedId = Crypt::decrypt($detailId);
                        } catch (\Exception $e) {
                            continue;
                        }

                        $detail = ProjectDetails::where('id', $decodedId)
                            ->where('project_id', $project_id)
                            ->first();

                        if ($detail) {
                            $detail->update([
                                'project_detail_title' => $title,
                                'project_detail_text'  => $text,
                            ]);
                        }
                    }
                    // INSERT NEW
                    else {
                        ProjectDetails::create([
                            'project_id'            => $project_id,
                            'position_order'           => $position++,
                            'project_detail_title'  => $title,
                            'project_detail_text'   => $text,
                        ]);
                    }
                }

                DB::commit();

                return redirect()
                    ->route('manage.project.details.contents', $request->project)
                    ->with('success', 'Project details saved successfully!');
            } catch (\Throwable $e) {
                DB::rollBack();

                Log::error('Project Details save failed', [
                    'error' => $e->getMessage(),
                    'file'  => $e->getFile(),
                    'line'  => $e->getLine(),
                ]);

                return back()
                    ->withInput()
                    ->with('error', 'Something went wrong while saving details.');
            }
        }

        return view('admin.project_other_details', [
            'projectId' => $request->project,
            'main_page'    => 'projects_management',
            'currentPage' => 'manage_projects',
        ]);
    }

    public function getProjectDetails(Request $request)
    {
        $project_id = Crypt::decrypt($request->project);
        $ad_details = ProjectDetails::select('id', 'project_detail_title', 'project_detail_text', 'status')->where('project_id', $project_id)->get();

        $all_details = $ad_details->map(function ($ad_detail) {
            return [
                'detail_id' => Crypt::encrypt($ad_detail->id),
                'detail_title' => $ad_detail->project_detail_title,
                'detail_text' => $ad_detail->project_detail_text,
            ];
        });

        return response()->json($all_details);
    }

    public function deleteProjectDetails(Request $request)
    {
        $detail_id = Crypt::decrypt($request->detail);
        $project_detail = ProjectWhyChoose::find($detail_id);
        if ($project_detail->delete()) {
            return 1;
        } else {
            return 0;
        }
    }
}
