<?php

namespace App\Http\Controllers;

use App\Models\Enquiries;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

class EnquiriesController extends Controller
{
    public function addEnquiry(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'first_name'        => 'required|string|max:255',
                'last_name'         => 'required|string|max:255',
                'phone'             => 'required|numeric',
                'email'             => 'required|email',
                'message'           => 'nullable|string',
            ], [
                'first_name.required' => 'First name is required.',
                'first_name.string'   => 'First name must be a valid string.',
                'first_name.max'      => 'First name cannot exceed 255 characters.',

                'last_name.required'  => 'Last name is required.',
                'last_name.string'    => 'Last name must be a valid string.',
                'last_name.max'       => 'Last name cannot exceed 255 characters.',

                'phone.required'      => 'Phone number is required.',
                'phone.numeric'       => 'Phone number must be numeric.',

                'email.required'      => 'Email is required.',
                'email.email'         => 'Email must be a valid email address.',

                'message.string'      => 'Message must be a valid string.',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => $validator->errors()->first(),
                    'errors'  => $validator->errors()
                ], 422);
            }

            $fromName = env('APP_NAME_FOR_MAIL', config('app.name', 'MyApp'));
            Mail::send('emails.send-enquiry', [
                'first_name'  => $request->first_name,
                'last_name'   => $request->last_name,
                'phone'       => $request->phone,
                'email'       => $request->email,
                'messageText'     => $request->message ?? '',
            ], function ($m) use ($fromName) {
                $m->to(env('COMPANY_MAIL'))
                    ->from(
                        env('MAIL_FROM_ADDRESS'),
                        $fromName
                    )
                    ->subject('New Enquiry ✅');
            });

            Mail::send([], [], function ($m) use ($request, $fromName) {
                $htmlContent = "
                    <html>
                        <body style='font-family: Arial, sans-serif; background-color:#f9f9f9; padding:20px;'>
                            <div style='max-width:600px; margin:auto; background:#fff; border-radius:8px; padding:20px; box-shadow:0 0 10px rgba(0,0,0,0.05);'>
                                <h2 style='color:#333;'>Thank You, {$request->first_name}!</h2>
                                <p style='font-size:16px; color:#555;'>
                                    We’ve received your enquiry and our team will get back to you shortly.
                                </p>
                                <p style='font-size:14px; color:#777;'>
                                    If you have any urgent questions, feel free to reply to this email.
                                </p>
                                <br>
                                <p style='font-size:14px; color:#999;'>Best regards,<br><strong>{$fromName}</strong></p>
                            </div>
                        </body>
                    </html>
                ";

                $m->to($request->email)
                    ->from(env('MAIL_FROM_ADDRESS'), env('APP_NAME_FOR_MAIL', config('app.name', 'MyApp')))
                    ->subject('✅ Thank You for Your Enquiry')
                    ->html($htmlContent);
            });

            $enquiry = new Enquiries();
            $enquiry->first_name = $request->first_name;
            $enquiry->last_name = $request->last_name;
            $enquiry->phone = $request->phone;
            $enquiry->email = $request->email;
            $enquiry->message = $request->message;
            $enquiry->save();

            return response()->json([
                'status' => 'success',
                'message' => 'Enquiry is saved Successfully!',
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        }
    }
}
