<?php

namespace App\Http\Controllers;

use App\Mail\ContactEnquiryMail;
use App\Models\Admin\Assignments;
use App\Models\Admin\Banners;
use App\Models\Admin\Blogs;
use App\Models\Admin\Categories;
use App\Models\Admin\Company;
use App\Models\Admin\Courses;
use App\Models\Admin\Faqs;
use App\Models\Admin\Features;
use App\Models\Admin\FreeDownloads;
use App\Models\Admin\GuessPapers;
use App\Models\Admin\Pages;
use App\Models\Admin\PageSections;
use App\Models\Admin\Partners;
use App\Models\Admin\Pricing;
use App\Models\Admin\Projects;
use App\Models\Admin\Subjects;
use App\Models\Admin\Testimonials;
use App\Models\CartItems;
use App\Models\ContactEnquiryData;
use App\Models\CustomerCards;
use App\Models\Customers;
use App\Models\HandwrittenData;
use App\Models\Invoices;
use App\Models\OrderInvoices;
use App\Models\OrderItems;
use App\Models\Orders;
use App\Models\Transactions;
use App\Services\InvoiceService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Illuminate\Contracts\Encryption\DecryptException;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class SitePagesController extends Controller
{
    private function getCustomerId(Request $request)
    {
        $token = $request->bearerToken();

        if (!$token) {
            return response()->json([
                'success' => false,
                'message' => 'Missing auth token'
            ], 401);
        }

        try {
            return Crypt::decrypt($token);
        } catch (\Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid auth token'
            ], 401);
        }
    }

    public function getPage(Request $request)
    {
        $page_id = $request->page_id;
        $this_page = Pages::where(["id" => $page_id, "status" => 'active'])->first();
        $page = [
            'page_name' => $this_page->page_name,
            'page_headline' => $this_page->page_headline,
            'breadcrumb_headline' => $this_page->breadcrumb_headline,
            'page_image' => asset($this_page->page_image),
            'page_description' => html_entity_decode(htmlspecialchars_decode($this_page->description)),
            'meta_title' => $this_page->meta_title,
            'meta_keywords' => $this_page->meta_keyword,
            'meta_description' => html_entity_decode(htmlspecialchars_decode($this_page->meta_description)),
        ];
        return response()->json($page, 200);
    }

    // public function getPage()
    // {
    //     $get_page = Pages::select('id', 'page_name')
    //     ->where('page_id', $pageId)
    //     ->where('parent_id', 0)
    //     ->where('status', 'active')
    //     ->orderBy('sort_order', 'asc')
    //     ->get()
    //     ->map(function ($section) {
    //         return [
    //             'id' => Crypt::encryptString($section->id),
    //             'page_name' => $section->page_name
    //         ];
    //     });
    // }

    public function getPageSections(Request $request)
    {
        $page_id = $request->page_id;
        $get_sections = PageSections::select('id', 'default_section_name', 'section_headline', 'section_image', 'description')->where('page_id', $page_id)->whereNull('parent_id')->where('status', 'active')->orderBy('position_order', 'asc')->get();
        $data = $get_sections->map(function ($section) {
            return [
                'id' => Crypt::encryptString($section->id),
                'default_section_name' => $section->default_section_name,
                'section_headline' => $section->section_headline,
                'section_img' => asset($section->section_image),
                'section_desc' => html_entity_decode(htmlspecialchars_decode($section->description))
            ];
        });

        return response()->json($data, 200);
    }

    public function getPageSection(Request $request)
    {
        $section_id = $request->section_id;
        $this_section = PageSections::select('default_section_name', 'section_headline', 'section_title', 'description', 'section_image', 'button_name')->where(["id" => $section_id, "status" => 'active'])->first();
        if ($this_section) {
            $section = [
                'default_section_name'  => $this_section->default_section_name,
                'section_headline' => $this_section->section_headline,
                'section_title' => $this_section->section_title,
                'section_image' => asset($this_section->section_image),
                'description'  => html_entity_decode(htmlspecialchars_decode($this_section->description)),
                'button_name' => $this_section->button_name
            ];
        }

        return response()->json($section, 200);
    }

    public function getPageSubsections(Request $request)
    {
        $page_id = $request->page_id;
        $section_id = $request->section_id;
        $parent_section = PageSections::select('section_title', 'section_headline', 'description', 'section_image', 'section_video')->where(["id" => $section_id, "status" => 'active'])->first();

        $getSubsections = PageSections::select('section_icon', 'section_image', 'section_headline', 'description')->where(["page_id" => $page_id, "parent_id" => $section_id, "status" => 'active'])->get();
        // if ($section_id == 13) {
        //     $data = $getSubsections->map(function ($section, $index) {
        //         return [
        //             'delay' => 0.2 + ($index + 0.1),
        //             'title' => $section->section_headline,
        //             'desc' => strip_tags(htmlspecialchars_decode($section->description)),
        //         ];
        //     });
        // } else {
        $data = $getSubsections->map(function ($section, $index) {
            return [
                'number' => '0' . ($index + 1),
                'delay' => 0.2 + ($index + 0.1),
                'sec_icon' => $section->section_icon,
                'icon' => asset($section->section_image),
                'title' => $section->section_headline,
                'desc' => html_entity_decode(html_entity_decode($section->description)),
            ];
        });
        // }

        return response()->json([
            'parent_section_title' => $parent_section->section_title,
            'parent_section_headline' => $parent_section->section_headline,
            'parent_section_description' => html_entity_decode(htmlspecialchars_decode($parent_section->description)),
            'parent_section_image' => asset($parent_section->section_image),
            'parent_section_video' => asset($parent_section->section_video),
            'subsections' => $data
        ], 200);
    }

    public function getBanners()
    {
        $banners = Banners::select('banner_headline', 'banner_title', 'description', 'button_name', 'other_button_name', 'banner_image')->where(["status" => 'active'])->orderBy('position_order')->get();
        $allBanners = $banners->map(function ($banner, $index) {
            return [
                'id' => 'slide-assignments-' . $index,
                'title' => $banner->banner_headline,
                'subtitle' => $banner->banner_title,
                'text' => htmlspecialchars_decode($banner->description),
                'images' => [
                    asset($banner->banner_image)
                ],
                'ctaPrimary' => [
                    'text' => $banner->button_name ?? "Buy Now",
                    'href' => '#'
                ],
                'ctaSecondary' => [
                    'text' => $banner->other_button_name ?? "Help Desk",
                    'href' => '#'
                ],
            ];
        });

        return response()->json($allBanners, 200);
    }

    public function getCompanyData()
    {
        $website = Company::select('alert_message', 'email', 'phone', 'whatsapp_phone', 'alternate_phone', 'location', 'facebook_url', 'x_url', 'linkedin_url', 'youtube_url', 'instagram_url', 'telegram_url', 'whatsapp_url', 'availability')->first();

        return response()->json([
            "alert_message" => $website->alert_message,
            "email" => $website->email,
            "phone" => $website->phone,
            "whatsapp_phone" => $website->whatsapp_phone,
            "alternate_phone" => $website->alternate_phone,
            "location" => html_entity_decode(htmlspecialchars_decode($website->location)),
            "facebook_url" => $website->facebook_url,
            "x_url" => $website->x_url,
            "linkedin_url" => $website->linkedin_url,
            "youtube_url" => $website->youtube_url,
            "instagram_url" => $website->instagram_url,
            "telegram_url" => $website->telegram_url,
            "whatsapp_url" => $website->whatsapp_url,
            "availability" => html_entity_decode(htmlspecialchars_decode($website->availability)),
        ], 200);
    }

    // public function getHeaderLinks()
    // {
    //     $pages_links = Pages::select('header_footer_name', 'client_page_urls')->whereIn('visibility', ['both', 'header'])->where('status', 'active')->get();

    //     $header_links = $pages_links->map(function ($pages_link) {
    //         return [
    //             'header_name' => $pages_link->header_footer_name,
    //             'header_link' => $pages_link->client_page_urls
    //         ];
    //     });

    //     return response()->json($header_links, 200);
    // }

    // public function getFooterLinks()
    // {
    //     $pages_links = Pages::select('header_footer_name', 'client_page_urls')->whereIn('visibility', ['both', 'footer'])->where('status', 'active')->get();

    //     $footer_links = $pages_links->map(function ($pages_link) {
    //         return [
    //             'footer_name' => $pages_link->header_footer_name,
    //             'footer_link' => $pages_link->client_page_urls
    //         ];
    //     });

    //     return response()->json($footer_links, 200);
    // }

    public function contactEnquiryDataSave(Request $request)
    {
        try {
            // -------------------------
            // 1. Field Validation
            // -------------------------
            $validator = Validator::make($request->all(), [
                'first_name'    => 'required|string',
                'last_name'     => 'nullable|string',
                'email'         => 'required|string',
                'phone'         => 'required|string',
                'message'       => 'nullable|string',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors'  => $validator->errors(),
                ], 422);
            }

            $item = $validator->validated();

            // -------------------------
            // 2. Save Record
            // -------------------------
            ContactEnquiryData::create([
                'first_name'    => $item['first_name'],
                'last_name'     => $item['last_name'],
                'email'         => $item['email'],
                'phone'         => $item['phone'],
                'message'       => htmlspecialchars($item['message'] ?? '', ENT_QUOTES),
            ]);

            // -------------------------
            // 3. Send Email
            // -------------------------
            Mail::to('kumargtesting@gmail.com')->send(new ContactEnquiryMail($item));
            // Mail::to('info@ignouexcelsius.com')->send(new ContactEnquiryMail($item));

            return response()->json([
                'success' => true,
            ]);
        } catch (\Throwable $e) {

            Log::error('Contact Enquiry Save Error', [
                'message' => $e->getMessage(),
                'trace'   => $e->getTraceAsString(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Backend Error: An unexpected error occurred while saving the data.',
            ], 500);
        }
    }

    public function getPartners()
    {
        $partners = Partners::select('partner_name', 'partner_image')->where(["status" => 'active'])->orderBy('position_order')->get();
        $allPartners = $partners->map(function ($partner) {
            return [
                'partner_name' => $partner->partner_name,
                'partner_image' => asset($partner->partner_image)
            ];
        });

        return response()->json($allPartners, 200);
    }

    public function getFeatures()
    {
        $features = Features::select('feature_headline', 'description', 'feature_image')->where(["status" => 'active'])->orderBy('position_order')->get();
        $allFeatures = $features->map(function ($feature) {
            return [
                'title' => $feature->feature_headline,
                'desc' => html_entity_decode(htmlspecialchars_decode($feature->description)),
                'img' => asset($feature->feature_image)
            ];
        });

        return response()->json($allFeatures, 200);
    }

    public function getFaqs()
    {
        $faqs = Faqs::select('question', 'answer')->where(["status" => 'active'])->orderBy('position_order')->get();
        $allFaqs = $faqs->map(function ($faq) {
            return [
                'question' => $faq->question,
                'answer' => html_entity_decode(htmlspecialchars_decode($faq->answer))
            ];
        });

        return response()->json($allFaqs, 200);
    }

    public function getTestimonials()
    {
        $testimonials = Testimonials::select('client_name', 'description', 'rating_quantity', 'client_designation', 'client_city', 'client_image')->where(["status" => 'active'])->orderBy('position_order')->get();
        $allTestimonials = $testimonials->map(function ($testimonial) {
            $stars = '';
            for ($i = 1; $i <= $testimonial->rating_quantity; $i++) {
                $stars .= '★';
            }

            return [
                'name' => $testimonial->client_name,
                'rating' => $stars,
                'designation' => $testimonial->client_designation,
                'location' => $testimonial->client_city,
                'description' => html_entity_decode(htmlspecialchars_decode($testimonial->description)),
                'image' => asset($testimonial->client_image)
            ];
        });

        return response()->json($allTestimonials, 200);
    }

    public function getPlans()
    {
        $plans = Pricing::select('id', 'plan_name', 'plan_category', 'plan_price', 'plan_price_base', 'short_description', 'description', 'plan_image')->where(["status" => 'active'])->orderBy('position_order')->get();
        $allPlans = $plans->map(function ($plan) {
            return [
                'id' => Crypt::encrypt($plan->id),
                'org' => $plan->id,
                'title' => $plan->plan_name,
                'price' => $plan->plan_price,
                'price_per_base' => "₹" . $plan->plan_price . "/" . ($plan->plan_price_base == 'month' ? 'Month' : 'Year'),
                'img' => asset($plan->plan_image),
                'popular' => ($plan->plan_category == 'popular' ? true : false),
                'description' => html_entity_decode(htmlspecialchars_decode($plan->short_description)),
                'features' => html_entity_decode(htmlspecialchars_decode($plan->description))
            ];
        });

        return response()->json($allPlans, 200);
    }

    public function getBlogs()
    {
        $blogs = Blogs::select('blog_headline', 'blog_url', 'blog_category', 'short_description', 'description', 'blog_image', 'post_date')->where(["status" => 'active'])->orderBy('position_order')->get();
        $allBlogs = $blogs->map(function ($blog, $index) {
            return [
                'id' => ($index + 1),
                'title' => $blog->blog_headline,
                'slug' => $blog->blog_url,
                'image' => asset($blog->blog_image),
                'category' => ucwords($blog->blog_category),
                'date' => date("F j, Y", strtotime($blog->post_date)),
                'short_description' => strip_tags(html_entity_decode(htmlspecialchars_decode($blog->short_description))),
                'content' => html_entity_decode(htmlspecialchars_decode($blog->description))
            ];
        });

        return response()->json($allBlogs, 200);
    }

    public function getCoverPages()
    {
        $freeDownloads = FreeDownloads::where(['file_type' => 'cover_page', 'status' => 'active'])->orderBy('position_order')->get();

        $formatted = $freeDownloads->map(function ($item, $index) {
            return [
                'id' => $index + 1,
                'file_name' => $item->file_name,
                'file' => asset($item->file),
                'file_on_top_banner' => (bool) $item->file_on_top_banner,
            ];
        });

        return response()->json([
            'top_banner_files' => $formatted->where('file_on_top_banner', true)->values(),
            'files' => $formatted->where('file_on_top_banner', false)->values(),
        ], 200);
    }

    public function getFreeDownloads()
    {
        $freeDownloads = FreeDownloads::where(['file_type' => 'free_download', 'status' => 'active'])->orderBy('position_order')->get();

        $formatted = $freeDownloads->map(function ($item, $index) {
            return [
                'id' => $index + 1,
                'file_name' => $item->file_name,
                'file' => asset($item->file),
                'file_on_top_banner' => (bool) $item->file_on_top_banner,
            ];
        });

        return response()->json([
            'top_banner_files' => $formatted->where('file_on_top_banner', true)->values(),
            'files' => $formatted->where('file_on_top_banner', false)->values(),
        ], 200);
    }

    public function getBlog(Request $request)
    {
        $blog_url = $request->blog_url;
        $this_blog = Blogs::select('blog_headline', 'blog_url', 'blog_category', 'short_description', 'description', 'blog_image', 'post_date')->where(["blog_url" => $blog_url, "status" => 'active'])->orderBy('position_order')->first();
        $blog = [
            'title' => $this_blog->blog_headline,
            'slug' => $this_blog->blog_url,
            'image' => asset($this_blog->blog_image),
            'category' => ucwords($this_blog->blog_category),
            'date' => date("F j, Y", strtotime($this_blog->post_date)),
            'content' => html_entity_decode(htmlspecialchars_decode($this_blog->description))
        ];

        return response()->json($blog, 200);
    }

    public function getAssignmentCategoriesCourses($assignment_type)
    {
        $assignment_categories = Categories::select('id', 'category_generated_id', 'category_label', 'category_icon')->where(['parent_id' => null, 'status' => 'active'])->get();
        $result = [];
        foreach ($assignment_categories as $assignment_category) {
            // $related_courses = Courses::select('course_name AS title', 'course_code AS code')->where(['category_id' => $assignment_category->id, 'status' => 'active'])->get();
            $related_courses = Courses::query()
                ->join('subjects', function ($join) {
                    $join->whereRaw("
                        CONCAT(',', REPLACE(REPLACE(subjects.course_ids, '[', ''), ']', ''), ',')
                        LIKE CONCAT('%,', courses.id, ',%')
                    ");
                })
                ->join('assignments', 'assignments.subject_id', '=', 'subjects.id')
                ->where('assignments.assignment_type', $assignment_type)
                ->where('courses.category_id', $assignment_category->id)
                ->where('courses.status', 'active')
                ->distinct()
                ->get(['courses.course_name AS title', 'courses.course_code AS code', 'courses.course_url AS course_slug']);

            $result[] = [
                'id'    => $assignment_category->category_generated_id,
                'label' => $assignment_category->category_label,
                'icon'  => $assignment_category->category_icon,
                'items' => $related_courses,
            ];
        }

        return response()->json([
            'success' => true,
            'categories' => $result
        ], 200);
    }

    public function getAssignmentCards()
    {
        $assignments = Assignments::query()
            ->join('subjects', 'subjects.id', '=', 'assignments.subject_id')
            ->join('courses', function ($join) {
                $join->whereRaw("
                    CONCAT(',', REPLACE(REPLACE(subjects.course_ids, '[', ''), ']', ''), ',')
                    LIKE CONCAT('%,', courses.id, ',%')
                ");
            })
            ->where(['assignments.assignment_type' => 'solved', 'assignments.status' => 'active'])
            ->distinct()
            ->limit(8)
            ->get([
                'assignments.assignment_code',
                'assignments.assignment_title',
                'assignments.assignment_image',
                'assignments.assignment_url',
                'courses.course_url'
            ]);

        $cards = [];

        foreach ($assignments as $assignment) {
            $cards[] = [
                'code'  => $assignment->assignment_code,
                'title' => $assignment->assignment_title,
                'img'   => $assignment->assignment_image
                    ? asset($assignment->assignment_image)
                    : null,
                'to'    => '/assignments/solved/'
                    . $assignment->course_url . '/'
                    . $assignment->assignment_url,
            ];
        }

        return response()->json([
            'success' => true,
            'cards'   => $cards
        ], 200);
    }

    public function getGuessPaperCategoriesCourses()
    {
        $guess_paper_categories = Categories::select('id', 'category_generated_id', 'category_label', 'category_icon')->where(['parent_id' => null, 'status' => 'active'])->get();
        $result = [];
        foreach ($guess_paper_categories as $guess_paper_category) {
            $related_courses = Courses::query()
                ->join('subjects', function ($join) {
                    $join->whereRaw("
                        CONCAT(',', REPLACE(REPLACE(subjects.course_ids, '[', ''), ']', ''), ',')
                        LIKE CONCAT('%,', courses.id, ',%')
                    ");
                })
                ->join('guess_papers', 'guess_papers.subject_id', '=', 'subjects.id')
                ->where('courses.category_id', $guess_paper_category->id)
                ->where('courses.status', 'active')
                ->distinct()
                ->get(['courses.course_name AS title', 'courses.course_code AS code', 'courses.course_url AS course_slug']);

            $result[] = [
                'id'    => $guess_paper_category->category_generated_id,
                'label' => $guess_paper_category->category_label,
                'icon'  => $guess_paper_category->category_icon,
                'items' => $related_courses,
            ];
        }

        return response()->json([
            'success' => true,
            'categories' => $result
        ], 200);
    }

    public function getProjectCategoriesCourses()
    {
        $project_categories = Categories::select('id', 'category_generated_id', 'category_label', 'category_icon')->where(['parent_id' => null, 'status' => 'active'])->get();
        $result = [];
        foreach ($project_categories as $project_category) {
            // $related_courses = Courses::select('course_name AS title', 'course_code AS code')->where(['category_id' => $project_category->id, 'status' => 'active'])->get();
            $related_courses = Courses::query()
                ->join('subjects', function ($join) {
                    $join->whereRaw("
                        CONCAT(',', REPLACE(REPLACE(subjects.course_ids, '[', ''), ']', ''), ',')
                        LIKE CONCAT('%,', courses.id, ',%')
                    ");
                })
                ->join('projects', 'projects.subject_id', '=', 'subjects.id')
                ->where('courses.category_id', $project_category->id)
                ->where('courses.status', 'active')
                ->distinct()
                ->get(['courses.course_name AS title', 'courses.course_code AS code', 'courses.course_url AS course_slug']);

            $result[] = [
                'id'    => $project_category->category_generated_id,
                'label' => $project_category->category_label,
                'icon'  => $project_category->category_icon,
                'items' => $related_courses,
            ];
        }

        return response()->json([
            'success' => true,
            'categories' => $result
        ], 200);
    }

    public function getProjectCards()
    {
        $projects = Projects::query()
            ->join('subjects', 'subjects.id', '=', 'projects.subject_id')
            ->join('courses', function ($join) {
                $join->whereRaw("
                    CONCAT(',', REPLACE(REPLACE(subjects.course_ids, '[', ''), ']', ''), ',')
                    LIKE CONCAT('%,', courses.id, ',%')
                ");
            })
            ->where('projects.status', 'active')
            ->distinct()
            ->limit(8)
            ->get([
                'projects.project_code',
                'projects.project_title',
                'projects.project_image',
                'projects.project_url',
                'courses.course_url' // course_url from the course
            ]);

        $cards = [];

        foreach ($projects as $project) {
            $cards[] = [
                'code'  => $project->project_code,
                'title' => $project->project_title,
                'img'   => $project->project_image
                    ? asset($project->project_image)
                    : null,
                'to'    => '/projects/'
                    . $project->course_url . '/'
                    . $project->project_url,
            ];
        }

        return response()->json([
            'success' => true,
            'cards'   => $cards
        ], 200);
    }

    public function getCourseAssignments(Request $request)
    {
        $type = $request->type;
        $course_url = $request->course_slug;
        $course = Courses::where(['course_url' => $course_url, 'status' => 'active'])->first();

        if (! $course) {
            return response()->json([
                'success' => false,
                'message' => 'Course not found'
            ], 404);
        }

        $courseId = $course->id;
        $related_assignments = Assignments::query()
            ->join('subjects as s', function ($join) use ($courseId) {
                $join->whereRaw("
                    CONCAT(',', REPLACE(REPLACE(s.course_ids, '[', ''), ']', ''), ',')
                    LIKE CONCAT('%,', ?, ',%')
                ", [$courseId]);
            })
            ->whereColumn('assignments.subject_id', 's.id')
            ->select(
                'assignments.assignment_code',
                'assignments.assignment_subject_name',
                'assignments.assignment_url',
                'assignments.maximum_retail_price',
                'assignments.selling_price',
                'assignments.assignment_image',
                'assignments.type_ids',
                'assignments.language_ids',
                'assignments.session_ids',
                'assignments.coupon_ids'
            )
            ->where('assignments.assignment_type', $type)
            ->orderBy('assignments.id', 'desc')
            ->get()
            ->append(['types', 'languages', 'sessions', 'coupons']);

        $allRelatedAssignments = $related_assignments->map(function ($rel_assignment) {
            $assignment_types = $rel_assignment->types->map(fn($type) => [
                "enc_id" => Crypt::encrypt($type->id),
                "type_name" => $type->type_name
            ])
                ->values();
            $assignment_languages = $rel_assignment->languages->map(fn($language) => [
                "enc_id" => Crypt::encrypt($language->id),
                "language_name" => $language->language_name
            ])
                ->values();
            $assignment_sessions = $rel_assignment->sessions->map(fn($session) => [
                "enc_id" => Crypt::encrypt($session->id),
                "session_name" => $session->session_name
            ])
                ->values();

            return [
                'code'                  => $rel_assignment->assignment_code,
                'title'                 => $rel_assignment->assignment_subject_name,
                'assignment_slug'       => $rel_assignment->assignment_url,

                'types'                 => $assignment_types,
                'medium'                => $assignment_languages,
                'sessions'              => $assignment_sessions,

                'price'                 => (float) $rel_assignment->maximum_retail_price,
                'sale'                  => (float) $rel_assignment->selling_price,
                'img'                   => asset($rel_assignment->assignment_image),
            ];
        });

        return response()->json([
            'success' => true,
            'course' => $course,
            'list' => $allRelatedAssignments
        ], 200);
    }

    public function getCourseGuessPapers(Request $request)
    {
        $course_url = $request->course_slug;
        $course = Courses::where(['course_url' => $course_url, 'status' => 'active'])->first();

        if (! $course) {
            return response()->json([
                'success' => false,
                'message' => 'Course not found'
            ], 404);
        }

        $courseId = $course->id;
        $related_guess_papers = GuessPapers::query()
            ->join('subjects as s', function ($join) use ($courseId) {
                $join->whereRaw("
                    CONCAT(',', REPLACE(REPLACE(s.course_ids, '[', ''), ']', ''), ',')
                    LIKE CONCAT('%,', ?, ',%')
                ", [$courseId]);
            })
            ->whereColumn('guess_papers.subject_id', 's.id')
            ->select(
                'guess_papers.guess_paper_code',
                'guess_papers.guess_paper_subject_name',
                'guess_papers.guess_paper_url',
                'guess_papers.maximum_retail_price',
                'guess_papers.selling_price',
                'guess_papers.guess_paper_image',
                'guess_papers.type_ids',
                'guess_papers.language_ids',
                'guess_papers.session_ids',
                'guess_papers.coupon_ids'
            )
            ->orderBy('guess_papers.id', 'desc')
            ->get()
            ->append(['types', 'languages', 'sessions', 'coupons']);

        $allRelatedGuessPapers = $related_guess_papers->map(function ($rel_guess_paper) {
            $guess_paper_types = $rel_guess_paper->types->map(fn($type) => [
                "enc_id" => Crypt::encrypt($type->id),
                "type_name" => $type->type_name
            ])
                ->values();
            $guess_paper_languages = $rel_guess_paper->languages->map(fn($language) => [
                "enc_id" => Crypt::encrypt($language->id),
                "language_name" => $language->language_name
            ])
                ->values();
            $guess_paper_sessions = $rel_guess_paper->sessions->map(fn($session) => [
                "enc_id" => Crypt::encrypt($session->id),
                "session_name" => $session->session_name
            ])
                ->values();

            return [
                'code'                  => $rel_guess_paper->guess_paper_code,
                'title'                 => $rel_guess_paper->guess_paper_subject_name,
                'guess_paper_slug'          => $rel_guess_paper->guess_paper_url,

                'types'                 => $guess_paper_types,
                'medium'                => $guess_paper_languages,
                'sessions'              => $guess_paper_sessions,

                'price'                 => (float) $rel_guess_paper->maximum_retail_price,
                'sale'                  => (float) $rel_guess_paper->selling_price,
                'img'                   => asset($rel_guess_paper->guess_paper_image),
            ];
        });

        return response()->json([
            'success' => true,
            'course' => $course,
            'list' => $allRelatedGuessPapers
        ], 200);
    }

    public function getCourseProjects(Request $request)
    {
        $course_url = $request->course_slug;
        $course = Courses::where(['course_url' => $course_url, 'status' => 'active'])->first();

        if (! $course) {
            return response()->json([
                'success' => false,
                'message' => 'Course not found'
            ], 404);
        }

        $courseId = $course->id;
        $related_projects = Projects::query()
            ->join('subjects as s', function ($join) use ($courseId) {
                $join->whereRaw("
                    CONCAT(',', REPLACE(REPLACE(s.course_ids, '[', ''), ']', ''), ',')
                    LIKE CONCAT('%,', ?, ',%')
                ", [$courseId]);
            })
            ->whereColumn('projects.subject_id', 's.id')
            ->select(
                'projects.project_code',
                'projects.project_subject_name',
                'projects.project_url',
                'projects.maximum_retail_price',
                'projects.selling_price',
                'projects.project_image',
                'projects.type_ids',
                'projects.language_ids',
                'projects.session_ids',
                'projects.coupon_ids'
            )
            ->orderBy('projects.id', 'desc')
            ->get()
            ->append(['types', 'languages', 'sessions', 'coupons']);

        $allRelatedProjects = $related_projects->map(function ($rel_project) {
            $project_types = $rel_project->types->map(fn($type) => [
                "enc_id" => Crypt::encrypt($type->id),
                "type_name" => $type->type_name
            ])
                ->values();
            $project_languages = $rel_project->languages->map(fn($language) => [
                "enc_id" => Crypt::encrypt($language->id),
                "language_name" => $language->language_name
            ])
                ->values();
            $project_sessions = $rel_project->sessions->map(fn($session) => [
                "enc_id" => Crypt::encrypt($session->id),
                "session_name" => $session->session_name
            ])
                ->values();

            return [
                'code'                  => $rel_project->project_code,
                'title'                 => $rel_project->project_subject_name,
                'project_slug'          => $rel_project->project_url,

                'types'                 => $project_types,
                'medium'                => $project_languages,
                'sessions'              => $project_sessions,

                'price'                 => (float) $rel_project->maximum_retail_price,
                'sale'                  => (float) $rel_project->selling_price,
                'img'                   => asset($rel_project->project_image),
            ];
        });

        return response()->json([
            'success' => true,
            'course' => $course,
            'list' => $allRelatedProjects
        ], 200);
    }

    public function getSingleCourseAssignment(Request $request)
    {
        $course_slug = $request->course_slug;

        $course = Courses::select('id')
            ->where([
                'course_url' => $course_slug,
                'status' => 'active'
            ])
            ->first();

        if (! $course) {
            return response()->json([
                'success' => false,
                'message' => 'Course not found'
            ], 404);
        }

        $assignment_slug = $request->assignment_slug;

        $assignment = Assignments::select(
            'id',
            'subject_id',
            'assignment_code',
            'assignment_title',
            'breadcrumb_headline',
            'maximum_retail_price',
            'selling_price',
            'discount',
            'additional_discount',
            'discount_offer_expiry',
            'assignment_expiry',
            'assignment_quantity',
            'assignment_image',
            'question_paper_text',
            'short_description',
            'download_info',
            'description',
            'question_paper_image',
            'why_choose_headline',
            'cover_page_info',
            'meta_title',
            'meta_keyword',
            'meta_description',
            'type_ids',
            'language_ids',
            'session_ids',
            'coupon_ids'
        )
            ->with('subject:id,course_ids,subject_name')
            ->where([
                'assignment_url' => $assignment_slug
            ])
            ->first()
            ->append(['types', 'languages', 'sessions', 'coupons']);

        if (! $assignment) {
            return response()->json([
                'success' => false,
                'message' => 'Assignment not found'
            ], 404);
        }

        // calculate discount %
        $discount = $assignment->discount;
        $additionalDiscount = $assignment->additional_discount;
        // $discount = 0;
        // if ($assignment->maximum_retail_price > 0) {
        //     $discount = round(
        //         (($assignment->maximum_retail_price - $assignment->selling_price) / $assignment->maximum_retail_price) * 100
        //     );
        // }

        $offerEndsOn = $assignment->discount_offer_expiry;

        $assignment_types = $assignment->types->map(fn($type) => [
            "enc_id" => Crypt::encrypt($type->id),
            "type_name" => $type->type_name
        ])
            ->values();
        $assignment_languages = $assignment->languages->map(fn($language) => [
            "enc_id" => Crypt::encrypt($language->id),
            "language_name" => $language->language_name
        ])
            ->values();
        $assignment_sessions = $assignment->sessions->map(fn($session) => [
            "enc_id" => Crypt::encrypt($session->id),
            "session_name" => $session->session_name
        ])
            ->values();

        $language_list = $assignment->languages
            ->pluck('language_name')
            ->implode(', ');

        $course_short_titles = collect($assignment->subject->courses)
            ->pluck('course_short_title')
            ->implode(', ');

        $productData = [
            "code"                  => $assignment->assignment_code,
            "title"                 => $assignment->assignment_title,
            "breadcrumbTitle"       => $assignment->breadcrumb_headline,
            "maxRetailPrice"        => (float) $assignment->maximum_retail_price,
            "salePrice"             => (float) $assignment->selling_price,
            "assignment_quantity"   => (int) $assignment->assignment_quantity,
            "discount"              => $discount,
            "extraDiscount"         => $additionalDiscount,
            "offerEndsOn"           => $offerEndsOn ? date('c', strtotime($offerEndsOn)) : null,
            "assignmentsValidity"   => date('d M Y', strtotime($assignment->assignment_expiry)),
            "image"                 => asset($assignment->assignment_image),

            "types"                 => $assignment_types,
            "medium"                => $assignment_languages,
            "sessions"              => $assignment_sessions,
            "coupons"               => $assignment->coupons->map(function ($coupon) {
                return [
                    "couponName"        => $coupon->coupon_name,
                    "couponCode"        => $coupon->coupon,
                    "couponText"        => $coupon->coupon_relevant_text,
                    "couponValidity"    => $coupon->coupon_validity,
                ];
            })->toArray(),

            // placeholders — you can later fill from DB
            "highlightHeadline"  => $assignment->why_choose_headline ?? "Why Choose Our Solved Assignment?",
            "highlights"         => $assignment->highlights,

            "question_paper_text"  => html_entity_decode(htmlspecialchars_decode($assignment->question_paper_text)),
            "short_description"  => html_entity_decode(htmlspecialchars_decode($assignment->short_description)),
            "downloadInfo"       => html_entity_decode(htmlspecialchars_decode($assignment->download_info)),
            "main_description"   => html_entity_decode(htmlspecialchars_decode($assignment->description)),
            "assignmentDetails"  => [
                [
                    "title" => "University",
                    "text"  => "IGNOU (Indira Gandhi National Open University)"
                ],
                [
                    "title" => "Title",
                    "text"  => $assignment->assignment_title
                ],
                [
                    "title" => "Language(s)",
                    "text"  => $language_list
                ],
                [
                    "title" => "Course(s)",
                    "text"  => $course_short_titles
                ],
                [
                    "title" => "Code",
                    "text"  => $assignment->assignment_code
                ],
                [
                    "title" => "Subject",
                    "text"  => $assignment->subject->subject_name
                ],
            ],

            "questionPaperImage" => asset($assignment->question_paper_image),
            "coverPageInfo"      => html_entity_decode(htmlspecialchars_decode($assignment->short_description)),
        ];

        return response()->json([
            "success" => true,
            "assignment" => $productData,
        ], 200);
    }

    public function getSingleCourseGuessPaper(Request $request)
    {
        $course_slug = $request->course_slug;

        $course = Courses::select('id')
            ->where([
                'course_url' => $course_slug,
                'status' => 'active'
            ])
            ->first();

        if (! $course) {
            return response()->json([
                'success' => false,
                'message' => 'Course not found'
            ], 404);
        }

        $guess_paper_slug = $request->guess_paper_slug;

        $guess_paper = GuessPapers::select(
            'id',
            'subject_id',
            'guess_paper_code',
            'guess_paper_title',
            'breadcrumb_headline',
            'maximum_retail_price',
            'selling_price',
            'discount',
            'additional_discount',
            'discount_offer_expiry',
            'guess_paper_expiry',
            'guess_paper_quantity',
            'guess_paper_image',
            'short_description',
            'download_info',
            'description',
            'question_paper_image',
            'why_choose_headline',
            'meta_title',
            'meta_keyword',
            'meta_description',
            'type_ids',
            'language_ids',
            'session_ids',
            'coupon_ids'
        )
            ->with('subject:id,course_ids,subject_name')
            ->where([
                'guess_paper_url' => $guess_paper_slug
            ])
            ->first()
            ->append(['types', 'languages', 'sessions', 'coupons']);

        if (! $guess_paper) {
            return response()->json([
                'success' => false,
                'message' => 'Guess Paper not found'
            ], 404);
        }

        // calculate discount %
        $discount = $guess_paper->discount;
        $additionalDiscount = $guess_paper->additional_discount;
        // $discount = 0;
        // if ($guess_paper->maximum_retail_price > 0) {
        //     $discount = round(
        //         (($guess_paper->maximum_retail_price - $guess_paper->selling_price) / $guess_paper->maximum_retail_price) * 100
        //     );
        // }

        $offerEndsOn = $guess_paper->discount_offer_expiry;

        $guess_paper_types = $guess_paper->types->map(fn($type) => [
            "enc_id" => Crypt::encrypt($type->id),
            "type_name" => $type->type_name
        ])
            ->values();
        $guess_paper_languages = $guess_paper->languages->map(fn($language) => [
            "enc_id" => Crypt::encrypt($language->id),
            "language_name" => $language->language_name
        ])
            ->values();
        $guess_paper_sessions = $guess_paper->sessions->map(fn($session) => [
            "enc_id" => Crypt::encrypt($session->id),
            "session_name" => $session->session_name
        ])
            ->values();

        $language_list = $guess_paper->languages
            ->pluck('language_name')
            ->implode(', ');

        $course_short_titles = collect($guess_paper->subject->courses)
            ->pluck('course_short_title')
            ->implode(', ');

        $productData = [
            "code"                  => $guess_paper->guess_paper_code,
            "title"                 => $guess_paper->guess_paper_title,
            "breadcrumbTitle"       => $guess_paper->breadcrumb_headline,
            "maxRetailPrice"        => (float) $guess_paper->maximum_retail_price,
            "salePrice"             => (float) $guess_paper->selling_price,
            "guess_paper_quantity"   => (int) $guess_paper->guess_paper_quantity,
            "discount"              => $discount,
            "extraDiscount"         => $additionalDiscount,
            "offerEndsOn"           => $offerEndsOn ? date('c', strtotime($offerEndsOn)) : null,
            "guessPapersValidity"   => date('d M Y', strtotime($guess_paper->guess_paper_expiry)),
            "image"                 => asset($guess_paper->guess_paper_image),

            "types"                 => $guess_paper_types,
            "medium"                => $guess_paper_languages,
            "sessions"              => $guess_paper_sessions,
            "coupons"               => $guess_paper->coupons->map(function ($coupon) {
                return [
                    "couponName"        => $coupon->coupon_name,
                    "couponCode"        => $coupon->coupon,
                    "couponText"        => $coupon->coupon_relevant_text,
                    "couponValidity"    => $coupon->coupon_validity,
                ];
            })->toArray(),

            // placeholders — you can later fill from DB
            "highlightHeadline"  => $guess_paper->why_choose_headline ?? "Why Choose Our Guess Paper?",
            "highlights"         => $guess_paper->highlights,

            "short_description"  => html_entity_decode(htmlspecialchars_decode($guess_paper->short_description)),
            "downloadInfo"       => html_entity_decode(htmlspecialchars_decode($guess_paper->download_info)),
            "main_description"   => html_entity_decode(htmlspecialchars_decode($guess_paper->description)),
            "guessPaperDetails"  => [
                [
                    "title" => "University",
                    "text"  => "IGNOU (Indira Gandhi National Open University)"
                ],
                [
                    "title" => "Title",
                    "text"  => $guess_paper->guess_paper_title
                ],
                [
                    "title" => "Language(s)",
                    "text"  => $language_list
                ],
                [
                    "title" => "Course(s)",
                    "text"  => $course_short_titles
                ],
                [
                    "title" => "Code",
                    "text"  => $guess_paper->guess_paper_code
                ],
                [
                    "title" => "Subject",
                    "text"  => $guess_paper->subject->subject_name
                ],
            ],

            "questionPaperImage" => asset($guess_paper->question_paper_image),
            "coverPageInfo"      => html_entity_decode(htmlspecialchars_decode($guess_paper->short_description)),
        ];

        return response()->json([
            "success" => true,
            "guess_paper" => $productData,
        ], 200);
    }

    public function getSingleCourseProject(Request $request)
    {
        $course_slug = $request->course_slug;

        $course = Courses::select('id')
            ->where([
                'course_url' => $course_slug,
                'status' => 'active'
            ])
            ->first();

        if (! $course) {
            return response()->json([
                'success' => false,
                'message' => 'Course not found'
            ], 404);
        }

        $project_slug = $request->project_slug;

        $project = Projects::select(
            'id',
            'subject_id',
            'topics',
            'project_code',
            'project_title',
            'breadcrumb_headline',
            'maximum_retail_price',
            'selling_price',
            'discount',
            'additional_discount',
            'discount_offer_expiry',
            'project_expiry',
            'project_quantity',
            'project_image',
            'short_description',
            'download_info',
            'description',
            'question_paper_image',
            'why_choose_headline',
            'cover_page_info',
            'meta_title',
            'meta_keyword',
            'meta_description',
            'type_ids',
            'language_ids',
            'session_ids',
            'coupon_ids'
        )
            ->with('subject:id,course_ids,subject_name')
            ->where([
                'project_url' => $project_slug
            ])
            ->first()
            ->append(['types', 'languages', 'sessions', 'coupons']);

        if (! $project) {
            return response()->json([
                'success' => false,
                'message' => 'Project not found'
            ], 404);
        }

        // calculate discount %
        $discount = $project->discount;
        $additionalDiscount = $project->additional_discount;
        // $discount = 0;
        // if ($project->maximum_retail_price > 0) {
        //     $discount = round(
        //         (($project->maximum_retail_price - $project->selling_price) / $project->maximum_retail_price) * 100
        //     );
        // }

        $offerEndsOn = $project->discount_offer_expiry;

        $project_types = $project->types->map(fn($type) => [
            "enc_id" => Crypt::encrypt($type->id),
            "type_name" => $type->type_name
        ])
            ->values();
        $project_languages = $project->languages->map(fn($language) => [
            "enc_id" => Crypt::encrypt($language->id),
            "language_name" => $language->language_name
        ])
            ->values();
        $project_sessions = $project->sessions->map(fn($session) => [
            "enc_id" => Crypt::encrypt($session->id),
            "session_name" => $session->session_name
        ])
            ->values();

        $language_list = $project->languages
            ->pluck('language_name')
            ->implode(', ');

        $course_short_titles = collect(optional($project->subject)->courses ?? [])
            ->pluck('course_short_titles')
            ->filter()
            ->implode(', ');

        $project_topics = !empty($project->topics) && is_array($project->topics) ? $project->topics : [];

        $productData = [
            "code"                  => $project->project_code,
            "title"                 => $project->project_title,
            "breadcrumbTitle"       => $project->breadcrumb_headline,
            "maxRetailPrice"        => (float) $project->maximum_retail_price,
            "salePrice"             => (float) $project->selling_price,
            "project_quantity"   => (int) $project->project_quantity,
            "discount"              => $discount,
            "extraDiscount"         => $additionalDiscount,
            "offerEndsOn"           => $offerEndsOn ? date('c', strtotime($offerEndsOn)) : null,
            "projectsValidity"   => date('d M Y', strtotime($project->project_expiry)),
            "image"                 => asset($project->project_image),

            "types"                 => $project_types,
            "medium"                => $project_languages,
            "sessions"              => $project_sessions,
            "coupons"               => $project->coupons->map(function ($coupon) {
                return [
                    "couponName"        => $coupon->coupon_name,
                    "couponCode"        => $coupon->coupon,
                    "couponText"        => $coupon->coupon_relevant_text,
                    "couponValidity"    => $coupon->coupon_validity,
                ];
            })->toArray(),

            // placeholders — you can later fill from DB
            "highlightHeadline"  => $project->why_choose_headline ?? "Why Choose Our Project?",
            "highlights"         => $project->highlights,

            "short_description"  => html_entity_decode(htmlspecialchars_decode($project->short_description)),
            "downloadInfo"       => html_entity_decode(htmlspecialchars_decode($project->download_info)),
            "main_description"   => html_entity_decode(htmlspecialchars_decode($project->description)),
            "project_topics"     => $project_topics,
            "projectDetails"  => [
                [
                    "title" => "University",
                    "text"  => "IGNOU (Indira Gandhi National Open University)"
                ],
                [
                    "title" => "Title",
                    "text"  => $project->project_title
                ],
                [
                    "title" => "Language(s)",
                    "text"  => $language_list
                ],
                [
                    "title" => "Degree(s)",
                    "text"  => $course_short_titles
                ],
                [
                    "title" => "Code",
                    "text"  => $project->project_code
                ],
                [
                    "title" => "Subject",
                    "text"  => $project->subject->subject_name
                ],
            ],

            "questionPaperImage" => asset($project->question_paper_image),
            "coverPageInfo"      => html_entity_decode(htmlspecialchars_decode($project->short_description)),
        ];

        return response()->json([
            "success" => true,
            "project" => $productData,
        ], 200);
    }

    // public function getHandwrittenData()
    // {
    //     // Step 1: Fetch all handwritten assignments with subjects
    //     $assignments = Assignments::query()
    //         ->with('subject:id,subject_name,course_ids') // include course_ids JSON
    //         ->where('assignment_type', 'handwritten')
    //         ->get()
    //         ->map(function ($assignment) {
    //             $assignment->encrypted_id = Crypt::encrypt($assignment->id);
    //             $assignment->assignment_img = asset($assignment->assignment_image);
    //             return $assignment;
    //         })
    //         ->append(['languages', 'sessions']); // include medium, sessions

    //     // Step 2: Collect unique subjects
    //     $subjects = $assignments
    //         ->pluck('subject')
    //         ->filter()
    //         ->unique('id')
    //         ->values()
    //         ->map(function ($subject) {
    //             $subject->encrypted_id = Crypt::encrypt($subject->id);
    //             return $subject;
    //         });

    //     // Step 3: Collect unique course IDs from subjects
    //     $courseIds = $subjects
    //         ->pluck('course_ids')
    //         ->filter()
    //         ->flatMap(function ($ids) {
    //             return is_array($ids) ? $ids : json_decode($ids, true);
    //         })
    //         ->filter()
    //         ->unique()
    //         ->values();

    //     // Step 4: Fetch unique courses
    //     $uniqueCourses = Courses::whereIn('id', $courseIds)
    //         ->where('status', 'active')
    //         ->get()
    //         ->map(function ($course) {
    //             $course->encrypted_id = Crypt::encrypt($course->id);
    //             return $course;
    //         });

    //     // Step 5: Fetch categories (degrees) that have at least one course with assignments
    //     $categoryIds = $uniqueCourses->pluck('category_id')->unique();

    //     $categories = Categories::whereIn('id', $categoryIds)
    //         ->whereNull('parent_id')
    //         ->where('status', 'active')
    //         ->get(['id', 'category_generated_id', 'category_label', 'category_icon']);

    //     // Step 6: Build nested tree
    //     $categoriesTree = $categories->map(function ($category) use ($uniqueCourses, $subjects, $assignments) {

    //         // Courses under this category
    //         $courses = $uniqueCourses->filter(function ($course) use ($category) {
    //             return $course->category_id == $category->id;
    //         })->values();

    //         // Map each course to its subjects
    //         $coursesWithSubjects = $courses->map(function ($course) use ($subjects, $assignments) {

    //             // Subjects connected to this course
    //             $courseSubjects = $subjects->filter(function ($subject) use ($course) {
    //                 $subjectCourseIds = is_array($subject->course_ids) ? $subject->course_ids : json_decode($subject->course_ids, true);
    //                 return in_array($course->id, $subjectCourseIds);
    //             })->values();

    //             // Map each subject to its assignments
    //             $subjectsWithAssignments = $courseSubjects->map(function ($subject) use ($assignments) {
    //                 $subjectAssignments = $assignments->filter(function ($assignment) use ($subject) {
    //                     return $assignment->subject_id == $subject->id;
    //                 })->values();

    //                 return [
    //                     'id'          => $subject->id,
    //                     'encrypted_id'   => $subject->encrypted_id,
    //                     'subject_name' => $subject->subject_name,
    //                     'assignments' => $subjectAssignments,
    //                 ];
    //             });

    //             return [
    //                 'id'                => $course->id,
    //                 'encrypted_id'      => $course->encrypted_id,
    //                 'title'             => $course->course_name,
    //                 'code'              => $course->course_code,
    //                 'course_url'        => $course->course_url,
    //                 'subjects'          => $subjectsWithAssignments,
    //             ];
    //         });

    //         return [
    //             'id'    => $category->category_generated_id,
    //             'label' => $category->category_label,
    //             'icon'  => $category->category_icon,
    //             'courses' => $coursesWithSubjects,
    //         ];
    //     });

    //     // Step 7: Return final tree
    //     return response()->json([
    //         'categories_tree' => $categoriesTree,
    //     ], 200);
    // }

    public function getHandwrittenData()
    {
        $assignments = Assignments::where('assignment_type', 'handwritten')
            ->select('id', 'subject_id', 'language_ids', 'session_ids')
            ->get()
            ->append(['languages', 'sessions']);

        $assignmentsBySubject = $assignments->groupBy('subject_id');

        $subjectIds = $assignmentsBySubject->keys()->values();

        $subjects = Subjects::whereIn('id', $subjectIds)
            ->select('id', 'subject_code', 'course_ids')
            ->get()
            ->map(function ($subject) use ($assignmentsBySubject) {
                $subjectAssignments = $assignmentsBySubject[$subject->id] ?? collect();

                $languages = $subjectAssignments
                    ->flatMap(fn($assignment) => $assignment->languages ?? [])
                    ->values();

                $sessions = $subjectAssignments
                    ->flatMap(fn($assignment) => $assignment->sessions ?? [])
                    ->values();

                return [
                    'id'            => $subject->id,
                    'encrypted_id'  => Crypt::encrypt($subject->id),
                    'subject_code'  => $subject->subject_code,
                    'course_ids'    => is_array($subject->course_ids)
                        ? $subject->course_ids
                        : json_decode($subject->course_ids, true),
                    'languages' => $languages,
                    'sessions'  => $sessions,
                ];
            });

        $courseIds = collect($subjects)
            ->pluck('course_ids')
            ->flatten()
            ->unique()
            ->values();

        $allLanguages = collect($subjects)
            ->flatMap(fn($s) => $s['languages'])
            ->unique('id')
            ->values();

        $allSessions = collect($subjects)
            ->flatMap(fn($s) => $s['sessions'])
            ->unique('id')
            ->values();

        $courses = Courses::whereIn('id', $courseIds)
            ->where('status', 'active')
            ->select('id', 'course_name', 'course_code', 'course_url', 'category_id')
            ->get()
            ->map(function ($course) {
                return [
                    'id'           => $course->id,
                    'encrypted_id' => Crypt::encrypt($course->id),
                    'title'        => $course->course_name,
                    'code'         => $course->course_code,
                    'course_url'   => $course->course_url,
                    'category_id'  => $course->category_id,
                ];
            });

        $categoryIds = collect($courses)->pluck('category_id')->unique();

        $categories = Categories::whereIn('id', $categoryIds)
            ->whereNull('parent_id')
            ->where('status', 'active')
            ->select('id', 'category_generated_id', 'category_label', 'category_icon')
            ->get();

        $categoriesTree = $categories->map(function ($category) use ($courses, $subjects) {

            $categoryCourses = collect($courses)
                ->where('category_id', $category->id)
                ->values()
                ->map(function ($course) use ($subjects) {

                    $courseSubjects = collect($subjects)
                        ->filter(function ($subject) use ($course) {
                            return in_array($course['id'], $subject['course_ids']);
                        })
                        ->map(function ($subject) {
                            return [
                                'encrypted_id' => $subject['encrypted_id'],
                                'subject_code' => $subject['subject_code'],
                            ];
                        })
                        ->values();

                    return [
                        'encrypted_id' => $course['encrypted_id'],
                        'title'        => $course['title'],
                        'code'         => $course['code'],
                        'course_url'   => $course['course_url'],
                        'subjects'     => $courseSubjects,
                    ];
                });

            return [
                'id'       => $category->category_generated_id,
                'label'    => $category->category_label,
                'icon'     => $category->category_icon,
                'courses'  => $categoryCourses,
            ];
        });

        return response()->json([
            'categories_tree' => $categoriesTree,
            'all_languages'   => $allLanguages,
            'all_sessions'    => $allSessions,
        ], 200);
    }

    public function handwrittenDataSave(Request $request)
    {
        try {
            // -------------------------
            // 1. Auth Token Validation
            // -------------------------
            $enc_token = $request->bearerToken();

            if (!$enc_token) {
                return response()->json([
                    'success' => false,
                    'message' => 'Missing auth token'
                ], 401);
            }

            try {
                $customer_id = Crypt::decrypt($enc_token);
            } catch (DecryptException $e) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid auth token'
                ], 401);
            }

            // -------------------------
            // 2. Field Validation
            // -------------------------
            $validator = Validator::make($request->all(), [
                'degree_cat_gen_id'   => 'required|string',
                'course'              => 'required|string',
                'subjects'            => 'required|string',
                'language'            => 'nullable|string',
                'session'             => 'nullable|string',
                'quantity'            => 'required|integer|min:1',
                'price'               => 'required|numeric|min:0',
                'assignment_method'   => 'required|string|max:100',
                'comments'            => 'nullable|string|max:2000',
                'question_paper_file' => 'nullable|file|mimes:pdf|max:51200', // 50MB
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors'  => $validator->errors(),
                ], 422);
            }

            $item = $validator->validated();

            // -------------------------
            // 3. Safe Decrypt IDs
            // -------------------------
            try {
                $course_id     = Crypt::decrypt($item['course']);
                // $subject_id    = Crypt::decrypt($item['subject']);
                $subject_codes    = $item['subjects'];
                // $assignment_id = Crypt::decrypt($item['assignment']);

                $language_id = !empty($item['language'])
                    ? Crypt::decrypt($item['language'])
                    : null;

                $session_id = !empty($item['session'])
                    ? Crypt::decrypt($item['session'])
                    : null;
            } catch (DecryptException $e) {
                return response()->json([
                    'success' => false,
                    'message' => 'One or more encrypted IDs are invalid',
                ], 400);
            }

            // -------------------------
            // 4. Handle File Upload
            // -------------------------
            $fileName = null;
            $filePath = null;

            if ($request->hasFile('question_paper_file')) {
                $path = 'handwritten/question_paper_files/';
                $filePath = $this->storeImage($request->file('question_paper_file'), $path);
                $fileName = basename($filePath);
            }

            // -------------------------
            // 5. Save Record
            // -------------------------
            $handwritten_data = HandwrittenData::create([
                'customer_id'               => $customer_id,
                'category_generated_id'     => $item['degree_cat_gen_id'],
                'course_id'                 => $course_id,
                'subject_codes'             => $subject_codes,
                // 'assignment_id'             => $assignment_id,
                'language_id'               => $language_id,
                'session_id'                => $session_id,
                'quantity'                  => (int) $item['quantity'],
                'price'                     => (float) $item['price'],
                'assignment_method'         => $item['assignment_method'],
                'file_name'                 => $fileName,
                'file_path'                 => $filePath,
                'comments'                  => htmlspecialchars($item['comments'] ?? '', ENT_QUOTES),
            ]);

            return response()->json([
                'success' => true,
                'id'      => Crypt::encrypt($handwritten_data->id),
            ]);
        } catch (\Throwable $e) {

            Log::error('Handwritten Save Error', [
                'message' => $e->getMessage(),
                'trace'   => $e->getTraceAsString(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Backend Error: An unexpected error occurred while saving the data.',
            ], 500);
        }
    }

    public function cartUserList(Request $request)
    {
        try {
            $enc_token = $request->bearerToken();
            if (!$enc_token) {
                return response()->json([
                    'success' => false,
                    'message' => 'Missing auth token'
                ], 401);
            }

            $customer_id = Crypt::decrypt($enc_token);
            $cartItems = CartItems::where('customer_id', $customer_id)
                ->with([
                    'assignment:id,assignment_title,selling_price,assignment_url,assignment_image',
                    'project:id,project_title,selling_price,project_url,project_image',
                    'guessPaper:id,guess_paper_title,selling_price,guess_paper_url,guess_paper_image',
                    'course:id,course_url',
                ])
                ->get();

            $items = $cartItems->map(function ($cart) {
                $isProject    = !is_null($cart->project_id);
                $isGuessPaper = !is_null($cart->guess_paper_id);

                if ($isProject) {
                    $itemData = $cart->project;
                    $name     = $itemData->project_title;
                    $img      = $itemData->project_image;
                    $itemCode = $itemData->project_url;
                } elseif ($isGuessPaper) {
                    $itemData = $cart->guessPaper;
                    $name     = $itemData->guess_paper_title;
                    $img      = $itemData->guess_paper_image;
                    $itemCode = $itemData->guess_paper_url;
                } else {
                    $itemData = $cart->assignment;
                    $name     = $itemData->assignment_title;
                    $img      = $itemData->assignment_image;
                    $itemCode = $itemData->assignment_url;
                }

                return [
                    'id'    => $cart->item_id,
                    'name'  => $name,
                    'price' => (float) $itemData->selling_price,
                    'qty'   => (int) $cart->quantity,
                    'img'   => asset($img),

                    'courseCode' => $cart->course->course_url,
                    'itemCode'   => $itemCode,

                    // re-encrypt for frontend
                    'selectedType'    => $cart->type_id ? Crypt::encrypt($cart->type_id) : null,
                    'selectedMedium'  => $cart->language_id ? Crypt::encrypt($cart->language_id) : null,
                    'selectedSession' => $cart->session_id ? Crypt::encrypt($cart->session_id) : null,
                ];
            });

            return response()->json([
                'success' => true,
                'items' => $items,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load cart',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function merge(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'items' => 'required|array|min:1'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => $validator->errors()->first(),
                ], 422);
            }

            $enc_token = $request->bearerToken();
            if (!$enc_token) {
                return response()->json([
                    'success' => false,
                    'message' => 'Missing auth token'
                ], 401);
            }

            $customer_id = Crypt::decrypt($enc_token);
            $items = $request->items;

            foreach ($items as $item) {

                if (!isset($item['id']) || !is_string($item['id'])) {
                    continue;
                }

                $isProject = str_contains($item['id'], 'project');
                $isGuessPaper = str_contains($item['id'], 'guess-paper');

                if ($isProject) {
                    $itemData = Projects::select('id', 'subject_id')->where('project_url', $item['itemCode'])->first();
                } elseif ($isGuessPaper) {
                    $itemData = GuessPapers::select('id', 'subject_id')->where('guess_paper_url', $item['itemCode'])->first();
                } else {
                    $itemData = Assignments::select('id', 'subject_id')->where('assignment_url', $item['itemCode'])->first();
                }

                if (!$itemData) {
                    continue;
                }

                $subject = Subjects::select('id')->find($itemData->subject_id);
                $course = Courses::select('id')->where('course_url', $item['courseCode'])->first();

                if (!$subject || !$course) {
                    continue;
                }

                $type_id = $item['selectedType'] ? Crypt::decrypt($item['selectedType']) : null;
                $language_id = $item['selectedMedium'] ? Crypt::decrypt($item['selectedMedium']) : null;
                $session_id = $item['selectedSession'] ? Crypt::decrypt($item['selectedSession']) : null;

                $cartItem = CartItems::firstOrCreate(
                    [
                        'customer_id'       => $customer_id,
                        'item_id'           => $item['id'],
                        'course_id'         => $course->id,
                        'subject_id'        => $subject->id,
                        'assignment_id'     => (!$isProject && !$isGuessPaper) ? $itemData->id : null,
                        'project_id'        => $isProject ? $itemData->id : null,
                        'guess_paper_id'    => $isGuessPaper ? $itemData->id : null,
                    ],
                    [
                        'type_id'     => $type_id,
                        'language_id' => $language_id,
                        'session_id'  => $session_id,
                        'quantity'    => 0,
                    ]
                );

                $cartItem->increment('quantity', (int) $item['qty']);
            }

            return response()->json([
                'success' => true,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'An unexpected error occurred while merging cart.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function add(Request $request)
    {
        try {
            $enc_token = $request->bearerToken();
            if (!$enc_token) {
                return response()->json([
                    'success' => false,
                    'message' => 'Missing auth token'
                ], 401);
            }

            $customer_id = Crypt::decrypt($enc_token);
            $item = $request->all();

            if (!isset($item['id']) || !is_string($item['id'])) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid item ID'
                ], 400);
            }

            $isProject = str_contains($item['id'], 'project');
            $isGuessPaper = str_contains($item['id'], 'guess-paper');

            if ($isProject) {
                $itemData = Projects::select('id', 'subject_id')->where('project_url', $item['itemCode'])->first();
                if (!$itemData) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Project not found'
                    ], 404);
                }
            } elseif ($isGuessPaper) {
                $itemData = GuessPapers::select('id', 'subject_id')->where('guess_paper_url', $item['itemCode'])->first();
                if (!$itemData) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Guess Paper not found'
                    ], 404);
                }
            } else {
                $itemData = Assignments::select('id', 'subject_id')->where('assignment_url', $item['itemCode'])->first();
                if (!$itemData) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Assignment not found'
                    ], 404);
                }
            }

            $subject = Subjects::select('id')->findOrFail($itemData->subject_id);
            if (!$subject) {
                return response()->json([
                    'success' => false,
                    'message' => 'Subject not found'
                ], 404);
            }

            $course = Courses::select('id')->where('course_url', $item['courseCode'])->first();
            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            $type_id = $item['selectedType'] ? Crypt::decrypt($item['selectedType']) : null;
            $language_id = $item['selectedMedium'] ? Crypt::decrypt($item['selectedMedium']) : null;
            $session_id = $item['selectedSession'] ? Crypt::decrypt($item['selectedSession']) : null;

            $cartItem = CartItems::firstOrCreate(
                [
                    'customer_id'       => $customer_id,
                    'item_id'           => $item['id'],
                    'course_id'         => $course->id,
                    'subject_id'        => $subject->id,
                    'assignment_id'     => (!$isProject && !$isGuessPaper) ? $itemData->id : null,
                    'project_id'        => $isProject ? $itemData->id : null,
                    'guess_paper_id'    => $isGuessPaper ? $itemData->id : null,
                ],
                [
                    'type_id'     => $type_id,
                    'language_id' => $language_id,
                    'session_id'  => $session_id,
                    'quantity'    => 0,
                ]
            );

            $cartItem->increment('quantity', (int) $item['qty']);

            return response()->json([
                'success' => true,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'An unexpected error occurred while verifying the link.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function remove(Request $request)
    {
        try {
            $enc_token = $request->bearerToken();
            if (!$enc_token) {
                return response()->json([
                    'success' => false,
                    'message' => 'Missing auth token'
                ], 401);
            }

            $item_id = $request->all();
            if (!$item_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Item Id is required.'
                ], 400);
            }

            $customer_id = Crypt::decrypt($enc_token);
            $item = CartItems::where(['customer_id' => $customer_id, 'item_id' => $item_id])->first();
            if (!$item) {
                return response()->json([
                    'success' => false,
                    'message' => 'Item not found'
                ], 404);
            }

            $item->delete();

            return response()->json([
                'success' => true,
                'message' => 'Item removed successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'An unexpected error occurred while verifying the link.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function update(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'item_id'  => 'required|string',
                'quantity' => 'required|integer|min:1',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'message' => $validator->errors()->first(),
                ], 422);
            }

            $enc_token = $request->bearerToken();
            if (!$enc_token) {
                return response()->json([
                    'success' => false,
                    'message' => 'Missing auth token'
                ], 401);
            }

            $customer_id = Crypt::decrypt($enc_token);
            $item_id = $request->item_id;
            if (!$item_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Item Id not found'
                ], 404);
            }

            $item_quantity = $request->quantity;
            if ($item_quantity === null || $item_quantity < 1) {
                return response()->json([
                    'success' => false,
                    'message' => 'Quantity must be at least 1.'
                ], 404);
            }

            $updated = CartItems::where('customer_id', $customer_id)
                ->where('item_id', $item_id)
                ->update([
                    'quantity' => (int) $item_quantity
                ]);

            if (!$updated) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cart item not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'Item updated successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'An unexpected error occurred while verifying the link.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function cartUserClear(Request $request)
    {
        try {
            $enc_token = $request->bearerToken();
            if (!$enc_token) {
                return response()->json([
                    'success' => false,
                    'message' => 'Missing auth token'
                ], 401);
            }

            $customer_id = Crypt::decrypt($enc_token);
            $deletedCount = CartItems::where('customer_id', $customer_id)->delete();

            if ($deletedCount === 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cart is already empty'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'Cart cleared successfully'
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'An unexpected error occurred while verifying the link.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function placeOrder(Request $request)
    {
        $validator = Validator::make($request->all(), [
            // Billing
            'billingDetails.firstName' => 'required|string|max:100',
            'billingDetails.lastName'  => 'required|string|max:100',
            'billingDetails.email'     => 'required|email',
            'billingDetails.phone'     => 'required|string|max:20',
            'billingDetails.address'   => 'required|string',
            'billingDetails.country'   => 'required|string|max:5',
            'billingDetails.state'     => 'required|string|max:50',
            'billingDetails.zip'       => 'required|string|max:10',

            // Shipping
            'shippingDetails.firstName' => 'required|string|max:100',
            'shippingDetails.lastName'  => 'required|string|max:100',
            'shippingDetails.address'   => 'required|string',
            'shippingDetails.country'   => 'required|string|max:5',
            'shippingDetails.state'     => 'required|string|max:50',
            'shippingDetails.zip'       => 'required|string|max:10',

            // Payment
            'paymentMethod' => ['required', Rule::in(['card', 'upi', 'cod'])],

            // Items
            'items' => 'required|array|min:1',
            'items.*.id'    => 'required|string',
            'items.*.price' => 'required|numeric|min:0',
            'items.*.qty'   => 'required|integer|min:1',
            'items.*.courseCode' => 'required|string',
            'items.*.itemCode'   => 'required|string',
            'items.*.subjectIds' => 'nullable|array',

            // Page Name
            'pageName'      => 'required|string',

            // Totals
            'subtotal' => 'required|numeric|min:0',
            'total'    => 'required|numeric|min:0',

            'orderSource' => ['required', Rule::in(['CART', 'BUY_NOW'])],
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => $validator->errors()->first(),
            ], 422);
        }

        if ($request->orderSource === 'BUY_NOW' && count($request->items) !== 1) {
            return response()->json([
                'success' => false,
                'message' => 'Buy Now order must contain exactly one item'
            ], 422);
        }

        $enc_token = $request->bearerToken();
        if (!$enc_token) {
            return response()->json([
                'success' => false,
                'message' => 'Missing auth token'
            ], 401);
        }

        $customer_id = Crypt::decrypt($enc_token);

        DB::beginTransaction();

        try {
            /** -------------------------
             * 1) Generate Order Number
             * ------------------------- */
            $lastOrderId = DB::table('orders')->lockForUpdate()->max('id') ?? 0;
            $nextOrderId = $lastOrderId + 1;
            $orderNumber = 'ORD-' . now()->year . '-' . str_pad($nextOrderId, 6, '0', STR_PAD_LEFT);
            $isCOD = $request->paymentMethod === 'cod';

            /** -------------------------
             * 2) Create Order
             * ------------------------- */
            $order = Orders::create([
                'order_source'    => $request->orderSource,
                'order_number'    => $orderNumber,
                'customer_id'     => $customer_id,
                'billing_details'  => $request->billingDetails,
                'shipping_details' => $request->shippingDetails,
                'payment_method'  => $request->paymentMethod,
                'subtotal'        => $request->subtotal,
                'total'           => $request->total,
                'page_name'        => $request->pageName,
                'order_status'    => $isCOD ? 'pending' : 'pending',
            ]);

            /** -------------------------
             * 3) Create Order Items
             * ------------------------- */
            if ($request->orderSource === "CART") {
                foreach ($request->items as $item) {
                    $itemId = $item['id'];

                    $cart_item = CartItems::where([
                        'customer_id' => $customer_id,
                        'item_id'     => $itemId
                    ])->first();

                    if (!$cart_item) {
                        throw new \Exception("Cart item not found: {$itemId}");
                    }

                    $course = Courses::select('id')
                        ->where('course_url', $item['courseCode'])
                        ->first();

                    if (!$course) {
                        throw new \Exception("Course not found: {$item['courseCode']}");
                    }

                    OrderItems::create([
                        'order_id'      => $order->id,
                        'course_id'     => $course->id,
                        'assignment_id' => $cart_item->assignment_id,
                        'project_id'    => $cart_item->project_id,
                        'price'         => $item['price'],
                        'quantity'      => $item['qty'],
                        'total'         => $item['price'] * $item['qty'],
                    ]);

                    CartItems::where('id', $cart_item->id)->delete();
                }
            } else {
                // BUY NOW
                $item = $request->items[0];

                $course = Courses::select('id')
                    ->where('course_url', $item['courseCode'])
                    ->first();

                if (!$course) {
                    throw new \Exception("Course not found: {$item['courseCode']}");
                }

                if ($request->pageName === "hard-copy-page") {
                    foreach ($item['subjectIds'] as $subject_code) {
                        // $subject_id = Crypt::decrypt($subjectId);

                        OrderItems::create([
                            'order_id'      => $order->id,
                            'course_id'     => $course->id,
                            'subject_code'    => $subject_code,
                            'price'         => $item['price'],
                            'quantity'      => $item['qty'],
                            'total'         => $item['price'] * $item['qty'],
                        ]);
                    }
                } else {
                    $isProject = str_contains($item['id'], 'project');
                    $isGuessPaper = str_contains($item['id'], 'guess-paper');

                    if ($isProject) {
                        $itemData = Projects::select('id')
                            ->where('project_url', $item['itemCode'])
                            ->first();
                    } elseif ($isGuessPaper) {
                        $itemData = GuessPapers::select('id')
                            ->where('guess_paper_url', $item['itemCode'])
                            ->first();
                    } else {
                        $itemData = Assignments::select('id')
                            ->where('assignment_url', $item['itemCode'])
                            ->first();
                    }

                    if (!$itemData) {
                        throw new \Exception("Item not found: {$item['itemCode']}");
                    }

                    OrderItems::create([
                        'order_id'          => $order->id,
                        'course_id'         => $course->id,
                        'assignment_id'     => (!$isProject && !$isGuessPaper) ? $itemData->id : null,
                        'project_id'        => $isProject ? $itemData->id : null,
                        'guess_paper_id'    => $isGuessPaper ? $itemData->id : null,
                        'price'             => $item['price'],
                        'quantity'          => $item['qty'],
                        'total'             => $item['price'] * $item['qty'],
                    ]);
                }
            }

            /** -------------------------
             * 4) Create Transaction
             * ------------------------- */
            $transaction = null;
            $paymentStatus = 'pending';
            if (!$isCOD) {
                $transaction = Transactions::create([
                    'order_id'        => $order->id,
                    'customer_id'     => $customer_id,
                    'amount'          => $request->total,
                    'payment_method' => $request->paymentMethod,
                    'status'          => 'initiated',
                ]);

                /** -------------------------
                 * 5) Process Payment
                 * ------------------------- */
                $paymentStatus = 'success';

                $txnRef = 'TXN-' . now()->year . '-' . str_pad($transaction->id, 6, '0', STR_PAD_LEFT);

                $transaction->update([
                    'gateway_transaction_id' => $txnRef,
                    'status'                 => $paymentStatus,
                ]);
            }

            /** -------------------------
             * 6) Update Order Status
             * ------------------------- */
            $orderStatus = match ($paymentStatus) {
                'success' => 'confirmed',
                'failed'  => 'failed',
                default   => 'pending',
            };

            $order->update([
                'order_status' => $orderStatus,
            ]);

            /** -------------------------
             * 7) Create Invoice
             * ------------------------- */
            $invoiceNumber = 'INV-' . now()->year . '-' . str_pad($order->id, 6, '0', STR_PAD_LEFT);

            $invoice = Invoices::create([
                'invoice_number' => $invoiceNumber,
                'customer_id'    => $customer_id,
                'amount'         => $request->total,
                'payment_mode'   => $request->paymentMethod,
                'status'         => $paymentStatus === 'success' ? 'paid' : 'unpaid',
            ]);

            InvoiceService::generateAndSend($customer_id, $order, $transaction, $invoice->id);

            /** -------------------------
             * 8) Map Order ↔ Invoice
             * ------------------------- */
            OrderInvoices::create([
                'order_id'      => $order->id,
                'invoice_id'    => $invoice->id,
                'transaction_id' => $transaction ? $transaction->id : null,
            ]);

            DB::commit();

            return response()->json([
                'success'        => true,
                'message'        => 'Order placed successfully',
                'order_number'  => $orderNumber,
                'invoice_number' => $invoiceNumber,
                'payment_status' => $paymentStatus,
            ], 201);
        } catch (\Throwable $e) {
            DB::rollBack();

            Log::error('Place order failed', [
                'error'   => $e->getMessage(),
                'trace'   => $e->getTraceAsString(),
                'request' => $request->all(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to place order',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function getCustomerPayments(Request $request)
    {
        try {
            /** -------------------------
             * Auth Token
             * ------------------------- */
            $enc_token = $request->bearerToken();
            if (!$enc_token) {
                return response()->json([
                    'success' => false,
                    'message' => 'Missing auth token'
                ], 401);
            }

            try {
                $customer_id = Crypt::decrypt($enc_token);
            } catch (\Throwable $e) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid auth token'
                ], 401);
            }

            /** -------------------------
             * Validate Query Params
             * ------------------------- */
            $validator = Validator::make($request->all(), [
                'order_status'    => 'nullable|in:pending,confirmed,failed',
                'payment_status'  => 'nullable|in:initiated,success,failed,pending',
                'payment_method'  => 'nullable|in:card,upi,cod',
                'from_date'       => 'nullable|date',
                'to_date'         => 'nullable|date',
                'sort_by'         => 'nullable|string',
                'sort_dir'        => 'nullable|in:asc,desc',
                'page'            => 'nullable|integer|min:1',
                'per_page'        => 'nullable|integer|min:1|max:100',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                ], 422);
            }

            /** -------------------------
             * Query
             * ------------------------- */
            $query = Orders::query()
                ->with(['transactions', 'invoices'])
                ->where('customer_id', $customer_id);

            /** -------------------------
             * Filters
             * ------------------------- */
            if ($request->filled('order_status')) {
                $query->where('order_status', $request->order_status);
            }

            if ($request->filled('payment_status')) {
                $query->whereHas('transactions', function ($q) use ($request) {
                    $q->where('status', $request->payment_status);
                });
            }

            if ($request->filled('payment_method')) {
                $query->where('payment_method', $request->payment_method);
            }

            if ($request->filled('from_date') && $request->filled('to_date')) {
                if ($request->from_date > $request->to_date) {
                    return response()->json([
                        'success' => false,
                        'message' => 'from_date must be earlier than to_date'
                    ], 422);
                }

                $query->whereBetween('created_at', [
                    $request->from_date . ' 00:00:00',
                    $request->to_date   . ' 23:59:59'
                ]);
            }

            /** -------------------------
             * Sorting (Whitelist)
             * ------------------------- */
            $allowedSortFields = ['created_at', 'total', 'order_number', 'order_status'];
            $sortBy  = $request->get('sort_by', 'created_at');
            $sortDir = $request->get('sort_dir', 'desc');

            if (!in_array($sortBy, $allowedSortFields)) {
                $sortBy = 'created_at';
            }

            if (!in_array($sortDir, ['asc', 'desc'])) {
                $sortDir = 'desc';
            }

            $query->orderBy($sortBy, $sortDir);

            /** -------------------------
             * Pagination
             * ------------------------- */
            $perPage  = $request->get('per_page', 20);
            $payments = $query->paginate($perPage);

            /** -------------------------
             * Transform Response
             * ------------------------- */
            $data = $payments->getCollection()->transform(function ($order) {

                $transaction = $order->transactions->sortByDesc('id')->first();
                $invoice     = $order->invoices->sortByDesc('id')->first();

                return [
                    'order_id'           => $order->id,
                    'order_number'       => $order->order_number,
                    'order_status'       => $order->order_status,
                    'payment_method'     => $order->payment_method,
                    'amount'             => $order->total,

                    'transaction_id'     => optional($transaction)->id,
                    'transaction_status' => optional($transaction)->status,
                    'gateway_txn_id'     => optional($transaction)->gateway_transaction_id,

                    'invoice_id'         => optional($invoice)->id,
                    'invoice_number'     => optional($invoice)->invoice_number,
                    'invoice_status'     => optional($invoice)->status,

                    'created_at'         => $order->created_at->toDateTimeString(),
                ];
            });

            return response()->json([
                'success'    => true,
                'message'    => 'Payments fetched successfully',
                'data'       => $data,
                'pagination' => [
                    'total'        => $payments->total(),
                    'per_page'     => $payments->perPage(),
                    'current_page' => $payments->currentPage(),
                    'last_page'    => $payments->lastPage(),
                ],
            ]);
        } catch (\Throwable $e) {

            Log::error('getCustomerPayments failed', [
                'error'   => $e->getMessage(),
                'trace'   => $e->getTraceAsString(),
                'request' => $request->all(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch payments'
            ], 500);
        }
    }

    public function getCustomerOrders(Request $request)
    {
        try {

            /** -------------------------
             * Auth Token
             * ------------------------- */
            $enc_token = $request->bearerToken();

            if (!$enc_token) {
                return response()->json([
                    'success' => false,
                    'message' => 'Missing auth token'
                ], 401);
            }

            try {
                $customer_id = Crypt::decrypt($enc_token);
            } catch (\Throwable $e) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid auth token'
                ], 401);
            }

            /** -------------------------
             * Fetch Orders
             * ------------------------- */
            $orders = Orders::where('customer_id', $customer_id)
                ->select(
                    'order_number',
                    'total',
                    'order_status',
                    'created_at'
                )
                ->orderBy('created_at', 'desc')
                ->get();

            return response()->json([
                'success' => true,
                'message' => 'Orders fetched successfully',
                'data' => $orders
            ]);
        } catch (\Throwable $e) {

            Log::error('getCustomerOrders failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch orders'
            ], 500);
        }
    }

    public function getCustomerCards(Request $request)
    {
        $customer_id = $this->getCustomerId($request);
        if ($customer_id instanceof \Illuminate\Http\JsonResponse) {
            return $customer_id;
        }

        try {
            $cards = CustomerCards::where('customer_id', $customer_id)
                ->orderByDesc('is_default')
                ->latest()
                ->get();

            $allCards = $cards->map(function ($card) {
                return [
                    'encrypted_id' => $card->encrypted_id,
                    'holder_name' => $card->card_holder_name,
                    'card_type' => $card->card_type,
                    'last4' => $card->last4,
                    'expiry_month' => $card->expiry_month,
                    'expiry_year' => $card->expiry_year,
                    'is_default' => $card->is_default,
                ];
            });

            return response()->json([
                'success' => true,
                'message' => 'Customer cards fetched successfully',
                'data' => $allCards
            ]);
        } catch (\Throwable $e) {

            Log::error('Fetch customer cards failed', [
                'customer_id' => $customer_id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Unable to fetch customer cards'
            ], 500);
        }
    }

    public function addCustomerCard(Request $request)
    {
        $customer_id = $this->getCustomerId($request);
        if ($customer_id instanceof \Illuminate\Http\JsonResponse) {
            return $customer_id;
        }

        // return response()->json([
        //     'success' => false,
        //     'message' => $request->all(),
        // ], 422);

        $validator = Validator::make($request->all(), [
            'holder_name' => 'required',
            'card_type' => 'required',
            'card_number' => 'required|min:12',
            'expiry_month' => 'required',
            'expiry_year' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        try {

            if ($request->is_default) {
                CustomerCards::where('customer_id', $customer_id)
                    ->update(['is_default' => false]);
            }

            $card = CustomerCards::create([
                'customer_id'       => $customer_id,
                'card_holder_name'  => $request->holder_name,
                'card_type'         => $request->card_type,
                // 'card_number'       => $request->card_number,
                'last4'             => substr($request->card_number, -4),
                'expiry_month'      => $request->expiry_month,
                // 'expiry_year'       => substr($request->expiry_year, -2),
                'expiry_year'       => $request->expiry_year,
                'is_default'        => $request->is_default ?? false,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Card added successfully',
                'data' => $card
            ], 201);
        } catch (\Throwable $e) {

            Log::error('Card store failed', ['error' => $e->getMessage()]);

            return response()->json([
                'success' => false,
                'message' => 'Unable to add card'
            ], 500);
        }
    }

    public function updateCustomerCard(Request $request, $encrypted_id)
    {
        $customer_id = $this->getCustomerId($request);
        if ($customer_id instanceof \Illuminate\Http\JsonResponse) {
            return $customer_id;
        }

        // return response()->json([
        //     'success' => false,
        //     'message' => $request->all(),
        // ], 422);

        $validator = Validator::make($request->all(), [
            'holder_name' => 'required',
            'card_type' => 'required',
            'card_number' => 'required|min:12',
            'expiry_month' => 'required',
            'expiry_year' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ], 422);
        }

        try {
            $id = Crypt::decrypt($encrypted_id);

            $card = CustomerCards::where('id', $id)
                ->where('customer_id', $customer_id)
                ->firstOrFail();

            if ($request->is_default) {
                CustomerCards::where('customer_id', $customer_id)
                    ->update(['is_default' => false]);
            }

            $card->update([
                'card_holder_name' => $request->holder_name ?? $card->card_holder_name,
                'card_type' => $request->card_type ?? $card->card_type,
                // 'card_number' => $request->card_number,
                'last4' => substr($request->card_number, -4),
                'expiry_month' => $request->expiry_month ?? $card->expiry_month,
                // 'expiry_year' => $request->expiry_year
                //     ? substr($request->expiry_year, -2)
                //     : $card->expiry_year,
                'expiry_year' => $request->expiry_year ?? $card->expiry_year,
                'is_default' => $request->is_default ?? $card->is_default,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Card updated successfully',
                'data' => $card
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {

            return response()->json([
                'success' => false,
                'message' => 'Card not found'
            ], 404);
        } catch (\Throwable $e) {

            Log::error('Card update failed', ['error' => $e->getMessage()]);

            return response()->json([
                'success' => false,
                'message' => 'Unable to update card'
            ], 500);
        }
    }

    public function deleteCustomerCard(Request $request, $encrypted_id)
    {
        $customer_id = $this->getCustomerId($request);
        if ($customer_id instanceof \Illuminate\Http\JsonResponse) {
            return $customer_id;
        }

        try {
            $id = Crypt::decrypt($encrypted_id);

            $card = CustomerCards::where('id', $id)
                ->where('customer_id', $customer_id)
                ->firstOrFail();

            $wasDefault = $card->is_default;

            $card->delete();

            if ($wasDefault) {
                CustomerCards::where('customer_id', $customer_id)
                    ->where('id', '!=', $id)
                    ->first()?->update(['is_default' => true]);
            }

            return response()->json([
                'success' => true,
                'message' => 'Card deleted successfully'
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {

            return response()->json([
                'success' => false,
                'message' => 'Card not found'
            ], 404);
        } catch (\Throwable $e) {

            Log::error('Card delete failed', ['error' => $e->getMessage()]);

            return response()->json([
                'success' => false,
                'message' => 'Unable to delete card'
            ], 500);
        }
    }

    public function setDefaultCard(Request $request, $encrypted_id)
    {
        $customer_id = $this->getCustomerId($request);
        if ($customer_id instanceof \Illuminate\Http\JsonResponse) {
            return $customer_id;
        }

        try {
            $id = Crypt::decrypt($encrypted_id);

            CustomerCards::where('customer_id', $customer_id)
                ->update(['is_default' => false]);

            $card = CustomerCards::where('id', $id)
                ->where('customer_id', $customer_id)
                ->firstOrFail();

            $card->update(['is_default' => true]);

            return response()->json([
                'success' => true,
                'message' => 'Default card updated',
                'data' => $card
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {

            return response()->json([
                'success' => false,
                'message' => 'Card not found'
            ], 404);
        } catch (\Throwable $e) {

            Log::error('Set default failed', ['error' => $e->getMessage()]);

            return response()->json([
                'success' => false,
                'message' => 'Unable to set default card'
            ], 500);
        }
    }
}
